/* account.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>

#include "intl.h"
#include "main.h"
#include "mainwindow.h"
#include "account.h"
#include "prefs.h"
#include "prefs_account.h"
#include "manage_window.h"
#include "gtkutils.h"
#include "utils.h"
#include "alertpanel.h"

typedef enum
{
	COL_DEFAULT	= 0,
	COL_NAME	= 1,
	COL_PROTOCOL	= 2,
	COL_SERVER	= 3
} EditAccountColumnPos;

# define N_EDIT_ACCOUNT_COLS	4

#define PREFSBUFSIZE		1024

PrefsAccount *cur_account;

static GList *account_list = NULL;

static struct EditAccount {
	GtkWidget *window;
	GtkWidget *clist;
	GtkWidget *close_btn;
} edit_account;

static void account_edit_create		(void);

static void account_edit_prefs		(void);
static void account_delete		(void);

static void account_up			(void);
static void account_down		(void);

static void account_set_default		(void);

static void account_edit_close		(void);
static void account_key_pressed		(GtkWidget	*widget,
					 GdkEventKey	*event,
					 gpointer	 data);

static gint account_clist_set_row	(PrefsAccount	*ac_prefs,
					 gint		 row);
static void account_clist_set		(void);

static void account_list_set		(void);

void account_read_config_all(void)
{
	GSList *ac_label_list = NULL, *cur;
	gchar *rcpath;
	FILE *fp;
	gchar buf[PREFSBUFSIZE];
	PrefsAccount *ac_prefs;

	debug_print(_("Reading all config for each account...\n"));

	rcpath = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, RC_DIR,
			     G_DIR_SEPARATOR_S, ACCOUNT_RC, NULL);
	if ((fp = fopen(rcpath, "r")) == NULL) {
		fprintf(stderr, "%s: ", rcpath);
		perror("fopen");
		g_free(rcpath);
		return;
	}
	g_free(rcpath);

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		if (!strncmp(buf, "[Account: ", 10)) {
			strretchomp(buf);
			memmove(buf, buf + 1, strlen(buf));
			buf[strlen(buf) - 1] = '\0';
			debug_print(_("Found label: %s\n"), buf);
			ac_label_list = g_slist_append(ac_label_list,
						       g_strdup(buf));
		}
	}
	fclose(fp);

	/* read config data from file */
	cur_account = NULL;
	for (cur = ac_label_list; cur != NULL; cur = cur->next) {
		ac_prefs = g_new0(PrefsAccount, 1);
		prefs_account_read_config(ac_prefs, (gchar *)cur->data);
		account_list = g_list_append(account_list, ac_prefs);
		if (ac_prefs->is_default)
			cur_account = ac_prefs;
	}
	/* if default is not set, assume first account as default */
	if (!cur_account && account_list) {
		ac_prefs = (PrefsAccount *)account_list->data;
		account_set_as_default(ac_prefs);
		cur_account = ac_prefs;
	}

	account_set_menu();
	main_window_show_cur_account();

	while (ac_label_list) {
		g_free(ac_label_list->data);
		ac_label_list = g_slist_remove(ac_label_list,
					       ac_label_list->data);
	}
}

void account_save_config_all(void)
{
	prefs_account_save_config_all(account_list);
}

void account_set_menu(void)
{
	main_window_set_account_menu(account_list);
}

void account_foreach(AccountFunc func, gpointer user_data)
{
	GList *cur;

	for (cur = account_list; cur != NULL; cur = cur->next)
		func((PrefsAccount *)cur->data, user_data);
}

void account_edit_open(void)
{
	debug_print(_("Opening account edit window...\n"));

	if (!edit_account.window)
		account_edit_create();

	account_clist_set();

	manage_window_set_transient(GTK_WINDOW(edit_account.window));
	gtk_widget_grab_focus(edit_account.close_btn);
	gtk_widget_show(edit_account.window);

	manage_window_focus_in(edit_account.window, NULL, NULL);
}

void account_add(void)
{
	PrefsAccount *ac_prefs;

	ac_prefs = prefs_account_open(NULL);

	if (!ac_prefs) return;

	account_list = g_list_append(account_list, ac_prefs);

	if (ac_prefs->is_default)
		account_set_as_default(ac_prefs);

	account_clist_set();
}

void account_set_as_default(PrefsAccount *ac_prefs)
{
	PrefsAccount *ap;
	GList *cur;

	for (cur = account_list; cur != NULL; cur = cur->next) {
		ap = (PrefsAccount *)cur->data;
		if (ap->is_default)
			ap->is_default = FALSE;
	}

	ac_prefs->is_default = TRUE;
}

static void account_edit_create(void)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *scrolledwin;
	GtkWidget *clist;
	gchar *titles[N_EDIT_ACCOUNT_COLS];

	GtkWidget *vbox2;
	GtkWidget *add_btn;
	GtkWidget *edit_btn;
	GtkWidget *del_btn;
	GtkWidget *up_btn;
	GtkWidget *down_btn;

	GtkWidget *default_btn;

	GtkWidget *hbbox;
	GtkWidget *close_btn;

	debug_print(_("Creating account edit window...\n"));

	window = gtk_window_new (GTK_WINDOW_DIALOG);
	gtk_widget_set_usize (window, 500, 320);
	gtk_container_set_border_width (GTK_CONTAINER (window), 8);
	gtk_window_set_title (GTK_WINDOW (window), _("Edit accounts"));
	gtk_window_set_modal (GTK_WINDOW (window), TRUE);
	gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			    GTK_SIGNAL_FUNC (account_edit_close), NULL);
	gtk_signal_connect (GTK_OBJECT (window), "key_press_event",
			    GTK_SIGNAL_FUNC (account_key_pressed), NULL);
	gtk_signal_connect (GTK_OBJECT (window), "focus_in_event",
			    GTK_SIGNAL_FUNC (manage_window_focus_in), NULL);

	vbox = gtk_vbox_new (FALSE, 12);
	gtk_widget_show (vbox);
	gtk_container_add (GTK_CONTAINER (window), vbox);

	hbox = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 2);

	scrolledwin = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwin);
	gtk_box_pack_start (GTK_BOX (hbox), scrolledwin, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);

	titles[COL_DEFAULT]  = "";
	titles[COL_NAME]     = _("Name");
	titles[COL_PROTOCOL] = _("Protocol");
	titles[COL_SERVER]   = _("Server");

	clist = gtk_clist_new_with_titles (N_EDIT_ACCOUNT_COLS, titles);
	gtk_widget_show (clist);
	gtk_container_add (GTK_CONTAINER (scrolledwin), clist);
	gtk_clist_set_column_width (GTK_CLIST(clist), COL_DEFAULT , 16);
	gtk_clist_set_column_width (GTK_CLIST(clist), COL_NAME    , 100);
	gtk_clist_set_column_width (GTK_CLIST(clist), COL_PROTOCOL, 70);
	gtk_clist_set_column_width (GTK_CLIST(clist), COL_SERVER  , 100);
	gtk_clist_set_selection_mode (GTK_CLIST(clist), GTK_SELECTION_BROWSE);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_box_pack_start (GTK_BOX (hbox), vbox2, FALSE, FALSE, 0);

	add_btn = gtk_button_new_with_label (_("Add"));
	gtk_widget_show (add_btn);
	gtk_box_pack_start (GTK_BOX (vbox2), add_btn, FALSE, FALSE, 4);
	gtk_signal_connect (GTK_OBJECT(add_btn), "clicked",
			    GTK_SIGNAL_FUNC (account_add), NULL);

	edit_btn = gtk_button_new_with_label (_("Edit"));
	gtk_widget_show (edit_btn);
	gtk_box_pack_start (GTK_BOX (vbox2), edit_btn, FALSE, FALSE, 4);
	gtk_signal_connect (GTK_OBJECT(edit_btn), "clicked",
			    GTK_SIGNAL_FUNC (account_edit_prefs), NULL);

	del_btn = gtk_button_new_with_label (_(" Delete "));
	gtk_widget_show (del_btn);
	gtk_box_pack_start (GTK_BOX (vbox2), del_btn, FALSE, FALSE, 4);
	gtk_signal_connect (GTK_OBJECT(del_btn), "clicked",
			    GTK_SIGNAL_FUNC (account_delete), NULL);

	down_btn = gtk_button_new_with_label (_("Down"));
	gtk_widget_show (down_btn);
	gtk_box_pack_end (GTK_BOX (vbox2), down_btn, FALSE, FALSE, 4);
	gtk_signal_connect (GTK_OBJECT(down_btn), "clicked",
			    GTK_SIGNAL_FUNC (account_down), NULL);

	up_btn = gtk_button_new_with_label (_("Up"));
	gtk_widget_show (up_btn);
	gtk_box_pack_end (GTK_BOX (vbox2), up_btn, FALSE, FALSE, 4);
	gtk_signal_connect (GTK_OBJECT(up_btn), "clicked",
			    GTK_SIGNAL_FUNC (account_up), NULL);

	hbox = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	vbox2 = gtk_vbox_new(FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_box_pack_start (GTK_BOX (hbox), vbox2, FALSE, FALSE, 0);

	default_btn = gtk_button_new_with_label (_(" Set as usually used account "));
	gtk_widget_show (default_btn);
	gtk_box_pack_start (GTK_BOX (vbox2), default_btn, TRUE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT(default_btn), "clicked",
			    GTK_SIGNAL_FUNC (account_set_default), NULL);

	gtkut_button_set_create(&hbbox, &close_btn, _("Close"),
				NULL, NULL, NULL, NULL);
	gtk_widget_show(hbbox);
	gtk_box_pack_end (GTK_BOX (hbox), hbbox, FALSE, FALSE, 0);
	gtk_widget_grab_default (close_btn);

	gtk_signal_connect (GTK_OBJECT (close_btn), "clicked",
			    GTK_SIGNAL_FUNC (account_edit_close),
			    NULL);

	edit_account.window    = window;
	edit_account.clist     = clist;
	edit_account.close_btn = close_btn;
}

static void account_edit_prefs(void)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	PrefsAccount *ac_prefs;
	gint row;
	gboolean prev_default;

	if (!clist->selection) return;

	row = GPOINTER_TO_INT(clist->selection->data);
	ac_prefs = gtk_clist_get_row_data(clist, row);
	prev_default = ac_prefs->is_default;
	prefs_account_open(ac_prefs);

	if (!prev_default && ac_prefs->is_default)
		account_set_as_default(ac_prefs);

	account_clist_set();
}

static void account_delete(void)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	PrefsAccount *ac_prefs;
	gint row;

	if (!clist->selection) return;

	if (alertpanel(_("Delete account"),
		       _("Do you really want to delete this account?"),
		       _("No"), _("Yes"), NULL) == G_ALERTDEFAULT)
		return;

	row = GPOINTER_TO_INT(clist->selection->data);
	ac_prefs = gtk_clist_get_row_data(clist, row);
	prefs_account_free(ac_prefs);

	gtk_clist_remove(clist, row);
	account_list = g_list_remove(account_list, ac_prefs);
}

static void account_up(void)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	gint row;

	if (!clist->selection) return;

	row = GPOINTER_TO_INT(clist->selection->data);
	if (row > 0) {
		gtk_clist_row_move(clist, row, row - 1);
		account_list_set();
	}
}

static void account_down(void)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	gint row;

	if (!clist->selection) return;

	row = GPOINTER_TO_INT(clist->selection->data);
	if (row < GPOINTER_TO_INT(clist->row_list_end->data)) {
		gtk_clist_row_move(clist, row, row + 1);
		account_list_set();
	}
}

static void account_set_default(void)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	gint row;
	PrefsAccount *ac_prefs;

	if (!clist->selection) return;

	row = GPOINTER_TO_INT(clist->selection->data);
	ac_prefs = gtk_clist_get_row_data(clist, row);
	account_set_as_default(ac_prefs);
	account_clist_set();
}

static void account_edit_close(void)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	gint row;

	/* set current account */
	if (clist->selection) {
		row = GPOINTER_TO_INT(clist->selection->data);
		cur_account = gtk_clist_get_row_data(clist, row);
	}

	account_list_set();
	account_save_config_all();

	main_window_set_account_menu(account_list);
	main_window_show_cur_account();

	gtk_widget_hide(edit_account.window);
}

static void account_key_pressed(GtkWidget *widget, GdkEventKey *event,
				gpointer data)
{
	if (event && event->keyval == GDK_Escape)
		account_edit_close();
}

/* set one CList row or add new row */
static gint account_clist_set_row(PrefsAccount *ac_prefs, gint row)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	gchar *text[N_EDIT_ACCOUNT_COLS];

	text[COL_DEFAULT] = ac_prefs->is_default ? "*" : "";
	text[COL_NAME] = ac_prefs->account_name;
	text[COL_PROTOCOL] = ac_prefs->protocol == A_POP3 ? "POP3" :
			     ac_prefs->protocol == A_APOP ? "APOP" : "";
	text[COL_SERVER] = ac_prefs->recv_server;

	if (row < 0)
		row = gtk_clist_append(clist, text);
	else {
		gtk_clist_set_text(clist, row, COL_DEFAULT, text[COL_DEFAULT]);
		gtk_clist_set_text(clist, row, COL_NAME, text[COL_NAME]);
		gtk_clist_set_text(clist, row, COL_PROTOCOL, text[COL_PROTOCOL]);
		gtk_clist_set_text(clist, row, COL_SERVER, text[COL_SERVER]);
	}

	gtk_clist_set_row_data(clist, row, ac_prefs);

	return row;
}

/* set CList from account list */
static void account_clist_set(void)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	GList *cur;
	gint prev_row;

	if (clist->selection)
		prev_row = GPOINTER_TO_INT(clist->selection->data);
	else
		prev_row = -1;

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);

	for (cur = account_list; cur != NULL; cur = cur->next) {
		gint row;

		row = account_clist_set_row((PrefsAccount *)cur->data, -1);
		if ((PrefsAccount *)cur->data == cur_account)
			gtk_clist_select_row(clist, row, -1);
	}

	if (prev_row >= 0)
		gtk_clist_select_row(clist, prev_row, -1);

	gtk_clist_thaw(clist);
}

/* set account list from CList */
static void account_list_set(void)
{
	GtkCList *clist = GTK_CLIST(edit_account.clist);
	gint row;
	PrefsAccount *ac_prefs;

	while (account_list)
		account_list = g_list_remove(account_list, account_list->data);

	for (row = 0; (ac_prefs = gtk_clist_get_row_data(clist, row)) != NULL;
	     row++)
		account_list = g_list_append(account_list, ac_prefs);
}
