/* compose.c */

#include <glib.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkitemfactory.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkctree.h>
#include <gtk/gtktext.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkeditable.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkhandlebox.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtktable.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkscrolledwindow.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/utsname.h>
#include <stdlib.h>
#include <wchar.h>
#include <wctype.h>
#include <kcc.h>

#include "main.h"
#include "mainwindow.h"
#include "compose.h"
#include "procmsg.h"
#include "menu.h"
#include "send.h"
#include "prefs_common.h"
#include "prefs_account.h"
#include "account.h"
#include "filesel.h"
#include "procheader.h"
#include "statusbar.h"
#include "about.h"
#include "base64.h"
#include "utils.h"
#include "alertpanel.h"

static Compose *compose_create			(void);
static void compose_toolbar_create		(Compose	*compose,
						 GtkWidget	*container);
static void compose_destroy			(Compose	*compose);

static void compose_reply_parse_header		(Compose	*compose,
						 MsgInfo	*msginfo,
						 FILE		*fp);
static void compose_quote_parse_fmt		(Compose	*compose,
						 MsgInfo	*msginfo);
static void compose_reply_set_entry		(Compose	*compose,
						 MsgInfo	*msginfo);
static void compose_insert_sig			(Compose	*compose);
static void compose_insert_file			(Compose	*compose,
						 const gchar	*file);

static gint compose_send			(Compose	*compose);
static gint compose_write_headers		(Compose	*compose,
						 FILE		*fp,
						 const gchar	*charset);

static void compose_convert_header		(gchar		*dest,
						 gint		 len,
						 gchar		*src,
						 gint		 header_len);
static gchar *tzoffset				(time_t		*now);
static void compose_get_date			(gchar		*buf,
						 gint		 len);
static void compose_generate_msgid		(gchar		*buf,
						 gint		 len);
static void compose_encode_header		(gchar		*dest,
						 gint		 len,
						 const gchar	*src,
						 gint		 header_len);

static GSList *compose_to_list_append		(GSList		*to_list,
						 const gchar	*str);
static GSList *compose_to_list_remove_all	(GSList		*to_list);

/* callback functions */

static void toolbar_send_cb		(GtkWidget	*widget,
					 gpointer	 data);
static void toolbar_insert_cb		(GtkWidget	*widget,
					 gpointer	 data);
static void toolbar_sig_cb		(GtkWidget	*widget,
					 gpointer	 data);

static void compose_send_cb		(gpointer	 data,
					 guint		 action,
					 GtkWidget	*widget);
static void compose_insert_file_cb	(gpointer	 data,
					 guint		 action,
					 GtkWidget	*widget);
static void compose_close_cb		(gpointer	 data,
					 guint		 action,
					 GtkWidget	*widget);

static void compose_cut_cb		(Compose	*compose);
static void compose_copy_cb		(Compose	*compose);
static void compose_paste_cb		(Compose	*compose);
static void compose_allsel_cb		(Compose	*compose);

static void compose_toggle_bcc_cb	(gpointer	 data,
					 guint		 action,
					 GtkWidget	*widget);

static void subject_activated		(GtkWidget	*widget,
					 Compose	*compose);
static void to_activated		(GtkWidget	*widget,
					 Compose	*compose);
static void cc_activated		(GtkWidget	*widget,
					 Compose	*compose);
static void bcc_activated		(GtkWidget	*widget,
					 Compose	*compose);

static GtkItemFactoryEntry compose_entries[] =
{
	{N_("/_File"),		NULL,		NULL,	0, "<Branch>"},
	{N_("/_File/_Attach file"), "<alt>A", NULL, 0, NULL},
	{N_("/_File/_Insert file"),
			"<control>I", compose_insert_file_cb, 0, NULL},
	{N_("/_File/Insert si_gnature"),
			"<control>G", compose_insert_sig, 0, NULL},
	{N_("/_File/---"),	NULL,		NULL,	0, "<Separator>"},
	{N_("/_File/_Close"),	"<alt>W",	compose_close_cb, 0, NULL},
	{N_("/_Edit"),		NULL,		NULL,	0, "<Branch>"},
	{N_("/_Edit/_Undo"),	"<control>Z",	NULL,	0, NULL},
	{N_("/_Edit/_Redo"),	"<control>Y",	NULL,	0, NULL},
	{N_("/_Edit/---"),	NULL,		NULL,	0, "<Separator>"},
	{N_("/_Edit/Cu_t"),	"<control>X",	compose_cut_cb, 0, NULL},
	{N_("/_Edit/_Copy"),	"<control>C",	compose_copy_cb, 0, NULL},
	{N_("/_Edit/_Paste"),	"<control>V",	compose_paste_cb, 0, NULL},
	{N_("/_Edit/Select _all"), "<control>A", compose_allsel_cb, 0, NULL},
	{N_("/_Message"),	NULL,		NULL,	0, "<Branch>"},
	{N_("/_Message/_Send"),	"<alt>S",	compose_send_cb, 0, NULL},
	{N_("/_Message/_Forward"),	"<control>F",	NULL,	0, NULL},
	{N_("/_Message/_Draft"),	"<alt>D",	NULL,	0, NULL},
	{N_("/_Message/---"),	NULL,		NULL,	0, "<Separator>"},
	{N_("/_Message/_Bcc"),
			NULL, compose_toggle_bcc_cb, 0, "<ToggleItem>"},
	{N_("/_Help"),		NULL,		NULL,	0, "<LastBranch>"},
	{N_("/_Help/_About"),	NULL,		about_show,	0, NULL}
};

static HeaderEntry hentry[] = {{"Reply-To: ",	NULL, TRUE},
			       {"Cc: ",		NULL, FALSE},
			       {"References: ",	NULL, FALSE},
			       {NULL,		NULL, FALSE}};

enum
{
	H_REPLY_TO	= 0,
	H_CC		= 1,
	H_REFERENCES	= 2
};

void compose_new(void)
{
	Compose *compose;

	compose = compose_create();

	if (prefs_common.auto_sig)
		compose_insert_sig(compose);
	gtk_editable_set_position(GTK_EDITABLE(compose->text), 0);

	gtk_widget_grab_focus(compose->subject_entry);
}

void compose_reply(MsgInfo *msginfo)
{
	Compose *compose;
	gchar *msgfile;
	FILE *fp;

	g_return_if_fail(msginfo != NULL);

	compose = compose_create();

	/* open parsing message */
	msgfile = g_strconcat(MSG_IS_NEWS(msginfo->flags) ?
			      get_news_cache_dir() : maildir,
			      G_DIR_SEPARATOR_S, msginfo->folder,
			      G_DIR_SEPARATOR_S, itos(msginfo->msgnum), NULL);
	fp = fopen(msgfile, "r");

	if (fp == NULL) {
		perror("fopen");
		g_warning(_("can't open message file %s\n"), msgfile);
		g_free(msgfile);
		return;
	}
	g_free(msgfile);

	compose_reply_parse_header(compose, msginfo, fp);
	fclose(fp);

	compose_reply_set_entry(compose, msginfo);

	if (prefs_common.auto_sig)
		compose_insert_sig(compose);
	gtk_editable_set_position(GTK_EDITABLE(compose->text), 0);

	gtk_widget_grab_focus(compose->text);
}

void compose_quote(MsgInfo *msginfo)
{
	Compose *compose;
	GtkText *text;
	FILE *fp;
	gchar buf[BUFFSIZE], buf2[BUFFSIZE];
	gchar *msgfile;
	gchar *qmark;

	g_return_if_fail(msginfo != NULL);

	compose = compose_create();

	/* open parsing and quoting message */
	msgfile = g_strconcat(MSG_IS_NEWS(msginfo->flags) ?
			      get_news_cache_dir() : maildir,
			      G_DIR_SEPARATOR_S, msginfo->folder,
			      G_DIR_SEPARATOR_S, itos(msginfo->msgnum), NULL);
	fp = fopen(msgfile, "r");

	if (fp == NULL) {
		perror("fopen");
		g_warning(_("can't open message file %s\n"), msgfile);
		g_free(msgfile);
		return;
	}
	g_free(msgfile);

	compose_reply_parse_header(compose, msginfo, fp);
	compose_reply_set_entry(compose, msginfo);

	text = GTK_TEXT(compose->text);
	gtk_text_freeze(text);

	compose_quote_parse_fmt(compose, msginfo);

	/* quote body */
	if (prefs_common.quotemark && *prefs_common.quotemark)
		qmark = prefs_common.quotemark;
	else
		qmark = "> ";
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		conv_jistoeuc(buf2, sizeof(buf2), buf);
		gtk_text_insert(text, NULL, NULL, NULL, qmark, -1);
		gtk_text_insert(text, NULL, NULL, NULL, buf2, -1);
	}
	fclose(fp);

	if (prefs_common.auto_sig)
		compose_insert_sig(compose);
	gtk_editable_set_position(GTK_EDITABLE(compose->text), 0);

	gtk_text_thaw(text);
	gtk_widget_grab_focus(compose->text);
}

void compose_forward(MsgInfo *msginfo)
{
	Compose *compose;
	GtkText *text;
	FILE *fp;
	gchar buf[BUFFSIZE], buf2[BUFFSIZE];
	gchar *msgfile;

	g_return_if_fail(msginfo != NULL);

	compose = compose_create();
	if (msginfo->subject && *msginfo->subject) {
		gtk_entry_set_text(GTK_ENTRY(compose->subject_entry), "Fwd: ");
		gtk_entry_append_text(GTK_ENTRY(compose->subject_entry),
				      msginfo->subject);
	}

	/* open quoting message */
	msgfile = g_strconcat(MSG_IS_NEWS(msginfo->flags) ?
			      get_news_cache_dir() : maildir,
			      G_DIR_SEPARATOR_S, msginfo->folder,
			      G_DIR_SEPARATOR_S, itos(msginfo->msgnum), NULL);
	fp = fopen(msgfile, "r");

	if (fp == NULL) {
		perror("fopen");
		g_warning(_("can't open message file %s\n"), msgfile);
		g_free(msgfile);
		return;
	}
	g_free(msgfile);

	/* skip header */
	while (procheader_get_one_field(buf, sizeof(buf), fp, NULL) != -1)
		;

	text = GTK_TEXT(compose->text);
	gtk_text_freeze(text);

	/* insert header */
	gtk_text_insert(text, NULL, NULL, NULL,
			_("\n\nBegin forwarded message:\n\n"), -1);
	if (msginfo->date && *msginfo->date) {
		gtk_text_insert(text, NULL, NULL, NULL,
				prefs_common.trans_hdr ? _("Date: ")
				: "Date: ", -1);
		gtk_text_insert(text, NULL, NULL, NULL, msginfo->date, -1);
		gtk_text_insert(text, NULL, NULL, NULL, "\n", 1);
	}
	if (msginfo->from && *msginfo->from) {
		gtk_text_insert(text, NULL, NULL, NULL,
				prefs_common.trans_hdr ? _("From: ")
				: "From: ", -1);
		gtk_text_insert(text, NULL, NULL, NULL, msginfo->from, -1);
		gtk_text_insert(text, NULL, NULL, NULL, "\n", 1);
	}
	if (msginfo->to && *msginfo->to) {
		gtk_text_insert(text, NULL, NULL, NULL,
				prefs_common.trans_hdr ? _("To: ")
				: "To: ", -1);
		gtk_text_insert(text, NULL, NULL, NULL, msginfo->to, -1);
		gtk_text_insert(text, NULL, NULL, NULL, "\n", 1);
	}
	if (msginfo->subject && *msginfo->subject) {
		gtk_text_insert(text, NULL, NULL, NULL,
				prefs_common.trans_hdr ? _("Subject: ")
				: "Subject: ", -1);
		gtk_text_insert(text, NULL, NULL, NULL, msginfo->subject, -1);
		gtk_text_insert(text, NULL, NULL, NULL, "\n\n\n", 3);
	}

	/* forward body */
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		conv_jistoeuc(buf2, sizeof(buf2), buf);
		gtk_text_insert(text, NULL, NULL, NULL, buf2, -1);
	}
	fclose(fp);

	if (prefs_common.auto_sig)
		compose_insert_sig(compose);
	gtk_editable_set_position(GTK_EDITABLE(compose->text), 0);

	gtk_text_thaw(text);
	gtk_widget_grab_focus(compose->to_entry);
}

static void compose_reply_parse_header(Compose *compose, MsgInfo *msginfo,
				       FILE *fp)
{
	procheader_get_header_fields(fp, hentry);

	if (hentry[H_REPLY_TO].body != NULL) {
		compose->replyto = hentry[H_REPLY_TO].body;
		hentry[H_REPLY_TO].body = NULL;
	}
	if (hentry[H_CC].body != NULL) {
		compose->cc = hentry[H_CC].body;
		hentry[H_CC].body = NULL;
	}
	if (hentry[H_REFERENCES].body != NULL) {
		if (msginfo->msgid && *msginfo->msgid) {
			compose->references =
				g_strconcat(hentry[H_REFERENCES].body,
					    "\n\t", "<", msginfo->msgid, ">",
					    NULL);
			g_free(hentry[H_REFERENCES].body);
		} else
			compose->references = hentry[H_REFERENCES].body;

		hentry[H_REFERENCES].body = NULL;
	}

	if (msginfo->msgid && *msginfo->msgid) {
		compose->inreplyto = g_strdup(msginfo->msgid);

		if (!compose->references) {
			if (msginfo->inreplyto && *msginfo->inreplyto)
				compose->references =
					g_strdup_printf("<%s>\n\t<%s>",
							msginfo->inreplyto,
							msginfo->msgid);
			else
				compose->references =
					g_strconcat("<", msginfo->msgid, ">",
						    NULL);
		}
	}
}

static void compose_quote_parse_fmt(Compose *compose, MsgInfo *msginfo)
{
	gchar *str;
	gchar *mbs;
	wchar_t *wcsfmt;
	wchar_t *sp;
	GtkText *text = GTK_TEXT(compose->text);

	if (!prefs_common.quotefmt || *prefs_common.quotefmt == '\0') return;

	wcsfmt = g_malloc((strlen(prefs_common.quotefmt) + 1) * sizeof(wchar_t));
	mbstowcs(wcsfmt, prefs_common.quotefmt, strlen(prefs_common.quotefmt) + 1);
	sp = wcsfmt;
	mbs = g_malloc(sizeof(wchar_t) + 1);

	while (*sp) {
		gint len;

		len = wctomb(mbs, *sp);
		mbs[len] = '\0';

		if (*mbs == '%') {
			wctomb(mbs, *(++sp));
			str = NULL;

			switch (*mbs) {
			case 'd':
				str = msginfo->date;
				break;
			case 'f':
				str = msginfo->from;
				break;
			case 's':
				str = msginfo->subject;
				break;
			case 't':
				str = msginfo->to;
				break;
			case 'i':
				str = g_strconcat("<", msginfo->msgid, ">",
						  NULL);
				gtk_text_insert(text, NULL, NULL, NULL,
						str, -1);
				g_free(str);
				str = NULL;
				sp++;
				break;
			case '%':
				str = "%";
				break;
			default:
				break;
			}

			if (str) {
				gtk_text_insert
					(text, NULL, NULL, NULL, str, -1);
				sp++;
			}
		} else if (*mbs == '\\') {
			wctomb(mbs, *(++sp));
			str = NULL;

			switch (*mbs) {
			case 'n':
				str = "\n";
				break;
			case 't':
				str = "\t";
				break;
			case '\\':
				str = "\\";
				break;
			default:
				break;
			}

			if (str) {
				gtk_text_insert
					(text, NULL, NULL, NULL, str, -1);
				sp++;
			}
		} else {
			gtk_text_insert(text, NULL, NULL, NULL, mbs, -1);
			sp++;
		}
	}

	g_free(mbs);
	g_free(wcsfmt);
}

static void compose_reply_set_entry(Compose *compose, MsgInfo *msginfo)
{
	gtk_entry_set_text(GTK_ENTRY(compose->to_entry),
			   compose->replyto ? compose->replyto
			   : msginfo->from);
	if (msginfo->subject && *msginfo->subject) {
		gchar *buf, *buf2, *p;

		buf = g_strdup(msginfo->subject);
		while (!strncasecmp(buf, "Re:", 3)) {
			p = buf + 3;
			while (isspace(*p)) p++;
			memmove(buf, p, strlen(p) + 1);
		}

		buf2 = g_strdup_printf("Re: %s", buf);
		gtk_entry_set_text(GTK_ENTRY(compose->subject_entry), buf2);
		g_free(buf2);
		g_free(buf);
	} else
		gtk_entry_set_text(GTK_ENTRY(compose->subject_entry), "Re: ");
}

static void compose_insert_sig(Compose *compose)
{
	gtk_text_insert(GTK_TEXT(compose->text), NULL, NULL, NULL,
			"\n---\n", -1);

	if (cur_account && cur_account->sig_path)
		compose_insert_file(compose, cur_account->sig_path);
	else {
		gchar *sigfile;

		sigfile = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S,
				      DEFAULT_SIGNATURE, NULL);
		compose_insert_file(compose, sigfile);
		g_free(sigfile);
	}
}

static void compose_insert_file(Compose *compose, const gchar *file)
{
	GtkText *text = GTK_TEXT(compose->text);
	gchar buf[BUFFSIZE];
	FILE *fp;

	g_return_if_fail(file != NULL);

	fp = fopen(file, "r");
	if (!fp) {
		perror("fopen");
		g_warning(_("can't open file %s\n"), file);
		return;
	}

	gtk_text_freeze(text);

	while (fgets(buf, sizeof(buf), fp) != NULL)
		gtk_text_insert(text, NULL, NULL, NULL, buf, -1);

	gtk_text_thaw(text);

	fclose(fp);
}

static gint compose_send(Compose *compose)
{
	size_t len;
	gchar tmpfile[22];
	FILE *fp;
	gchar *chars;
	gchar *buf;
	gchar *charset;
	gint ok;

	g_return_val_if_fail(cur_account != NULL, -1);

	buf = gtk_entry_get_text(GTK_ENTRY(compose->to_entry));
	if (*buf == '\0') {
		alertpanel(_("Error"), _("Receiptor is not specified.\n"),
			   NULL, NULL, NULL);
		return 1;
	}

	/* write to temporary file */
	g_snprintf(tmpfile, 22, "%s%ctmpmsg%d",
		   g_get_tmp_dir(), G_DIR_SEPARATOR, (gint)compose);
	if ((fp = fopen(tmpfile, "w")) != NULL) {
		/* chmod for security */
		if (chmod(tmpfile, S_IRUSR|S_IWUSR) < 0) {
			perror("chmod");
			g_warning(_("can't change file mode\n"));
		}

		/* get all composed text */
		chars = gtk_editable_get_chars(GTK_EDITABLE(compose->text),
					       0, -1);
		if (KCC_check(chars, 0) == ASCII) {
			buf = chars;
			chars = NULL;
			charset = "US-ASCII";
		} else {
			len = (strlen(chars) + 1) * 3;
			buf = g_malloc(len);
			conv_euctojis(buf, len, chars);
			charset = OUT_CHARSET;
		}

		/* write headers */
		if (compose_write_headers(compose, fp, charset) < 0) {
			g_warning(_("can't write headers\n"));
			fclose(fp);
			g_free(chars);
			g_free(buf);
			if (unlink(tmpfile) < 0)
				perror("unlink");
			return -1;
		}

		len = strlen(buf);
		if (fwrite(buf, sizeof(gchar), len, fp) != len) {
			g_warning(_("can't write to temporary file.\n"));
			fclose(fp);
			g_free(chars);
			g_free(buf);
			if (unlink(tmpfile) < 0)
				perror("unlink");
			return -1;
		} else {
			fclose(fp);
			g_free(chars);
			g_free(buf);
		}
	} else {
		perror("fopen");
		return -1;
	}

	ok = send_message_smtp(compose->to_list, tmpfile);
	statusbar_pop_all();

	/* save message to outbox */
	if (ok == 0 && prefs_common.savemsg) {
		FILE *fp;
		gint num;

		debug_print(_("saving sent message...\n"));
		if ((num = procmsg_copy_message(OUTBOX_DIR, tmpfile)) < 0)
			g_warning(_("can't save message\n"));
		else if ((fp = procmsg_open_mark_file(OUTBOX_DIR, TRUE))
			 == NULL)
			g_warning(_("can't open mark file\n"));
		else {
			MsgInfo newmsginfo;

			newmsginfo.msgnum = num;
			newmsginfo.flags = 0;
			procmsg_write_flags(&newmsginfo, fp);
			fclose(fp);
		}
	}

	/* queue message if failed to send */
	if (ok < 0 && prefs_common.queue_msg) {
		FILE *fp;
		gint num;

		debug_print(_("queueing message that failed to send...\n"));
		if ((num = procmsg_copy_message(QUEUE_DIR, tmpfile)) < 0)
			g_warning(_("can't queue message\n"));
		else if ((fp = procmsg_open_mark_file(QUEUE_DIR, TRUE))
			 == NULL)
			g_warning(_("can't open mark file\n"));
		else {
			MsgInfo newmsginfo;

			newmsginfo.msgnum = num;
			newmsginfo.flags = 0;
			procmsg_write_flags(&newmsginfo, fp);
			fclose(fp);
		}
	}

	if (unlink(tmpfile) < 0)
		perror("unlink");

	return ok;
}

static gint compose_write_headers(Compose *compose, FILE *fp,
				  const gchar *charset)
{
	gchar buf[BUFFSIZE];
	gchar *str;
	struct utsname utsbuf;

	g_return_val_if_fail(fp != NULL, -1);
	g_return_val_if_fail(cur_account != NULL, -1);
	g_return_val_if_fail(cur_account->address != NULL, -1);

	/* Date */
	if (cur_account->add_date) {
		compose_get_date(buf, sizeof(buf));
		fprintf(fp, "Date: %s\n", buf);
	}

	/* From */
	if (cur_account->name && *cur_account->name) {
		compose_convert_header(buf, sizeof(buf), cur_account->name,
				       strlen("From: "));
		fprintf(fp, "From: %s <%s>\n", buf, cur_account->address);
	} else
		fprintf(fp, "From: %s\n", cur_account->address);

	compose->to_list = compose_to_list_remove_all(compose->to_list);

	/* To */
	str = gtk_entry_get_text(GTK_ENTRY(compose->to_entry));
	if (*str != '\0') {
		str = g_strdup(str);
		g_strstrip(str);
		if (*str != '\0') {
			compose->to_list =
				compose_to_list_append(compose->to_list, str);
			compose_convert_header(buf, sizeof(buf), str,
					       strlen("To: "));
			fprintf(fp, "To: %s\n", buf);
			g_free(str);
		} else {
			g_free(str);
			return -1;
		}
	} else {
		g_free(str);
		return -1;
	}

	/* Cc */
	str = gtk_entry_get_text(GTK_ENTRY(compose->cc_entry));
	if (*str != '\0') {
		str = g_strdup(str);
		g_strstrip(str);
		if (*str != '\0') {
			compose->to_list =
				compose_to_list_append(compose->to_list, str);
			compose_convert_header(buf, sizeof(buf), str,
					       strlen("Cc: "));
			fprintf(fp, "Cc: %s\n", buf);
		}
		g_free(str);
	}

	/* Bcc */
	if (compose->use_bcc) {
		str = gtk_entry_get_text(GTK_ENTRY(compose->bcc_entry));
		if (*str != '\0') {
			str = g_strdup(str);
			g_strstrip(str);
			if (*str != '\0') {
				compose->to_list = compose_to_list_append
					(compose->to_list, str);
#if 0
				compose_convert_header(buf, sizeof(buf), str,
						       strlen("Bcc: "));
				fprintf(fp, "Bcc: %s\n", buf);
#endif
			}
			g_free(str);
		}
	}

	/* Subject */
	str = gtk_entry_get_text(GTK_ENTRY(compose->subject_entry));
	if (*str != '\0') {
		str = g_strdup(str);
		g_strstrip(str);
		if (*str != '\0') {
			compose_convert_header(buf, sizeof(buf), str,
					       strlen("Subject: "));
			fprintf(fp, "Subject: %s\n", buf);
		}
		g_free(str);
	}

	/* Message-ID */
	if (cur_account->gen_msgid) {
		compose_generate_msgid(buf, sizeof(buf));
		fprintf(fp, "Message-Id: <%s>\n", buf);
	}

	/* In-Reply-To */
	if (compose->inreplyto)
		fprintf(fp, "In-Reply-To: <%s>\n", compose->inreplyto);

	/* References */
	if (compose->references)
		fprintf(fp, "References: %s\n", compose->references);

	/* Program version and system info */
	uname(&utsbuf);
	fprintf(fp, "X-Mailer: %s (GTK+ %d.%d.%d; %s %s; %s)\n",
		prog_version,
		gtk_major_version, gtk_minor_version, gtk_micro_version,
		utsbuf.sysname, utsbuf.release, utsbuf.machine);

	/* Organization */
	if (cur_account->organization && *cur_account->organization) {
		compose_convert_header(buf, sizeof(buf),
				       cur_account->organization,
				       strlen("Organization: "));
		fprintf(fp, "Organization: %s\n", buf);
	}

	/* MIME */
	fprintf(fp, "Mime-Version: 1.0\n");
	fprintf(fp, "Content-Transfer-Encoding: 7bit\n");
	fprintf(fp, "Content-Type: text/plain; charset=%s\n",
		(charset && *charset) ? charset : OUT_CHARSET);

	/* separator between header and body */
	fputs("\n", fp);

	return 0;
}

static void compose_convert_header(gchar *dest, gint len, gchar *src,
				   gint header_len)
{
	if (len < 1) return;

	if (KCC_check(src, 0) == ASCII) {
		strncpy(dest, src, len);
		dest[len - 1] = '\0';
		return;
	} else
		compose_encode_header(dest, len, src, header_len);
}

/* calculate timezone offset */
static gchar *tzoffset(time_t *now)
{
	static gchar offset_string[6];
	struct tm gmt, *lt;
	gint off;
	gchar sign = '+';

	gmt = *gmtime(now);
	lt = localtime(now);
	off = (lt->tm_hour - gmt.tm_hour) * 60 + lt->tm_min - gmt.tm_min;
	if (lt->tm_year < gmt.tm_year)
		off -= 24 * 60;
	else if (lt->tm_year > gmt.tm_year)
		off += 24 * 60;
	else if (lt->tm_yday < gmt.tm_yday)
		off -= 24 * 60;
	else if (lt->tm_yday > gmt.tm_yday)
		off += 24 * 60;
	if (off < 0) {
		sign = '-';
		off = -off;
	}
	if (off >= 24 * 60)		/* should be impossible */
		off = 23 * 60 + 59;	/* if not, insert silly value */
	sprintf(offset_string, "%c%02d%02d", sign, off / 60, off % 60);
	return (offset_string);
}

static void compose_get_date(gchar *buf, gint len)
{
	struct tm *lt;
	time_t t;
	gchar day[4], mon[4];
	gint dd, hh, mm, ss, yyyy;

	t = time(NULL);
	lt = localtime(&t);

	sscanf(asctime(lt), "%s %s %d %d:%d:%d %d\n",
	       day, mon, &dd, &hh, &mm, &ss, &yyyy);
	g_snprintf(buf, len, "%s, %d %s %d %02d:%02d:%02d %s",
		   day, dd, mon, yyyy, hh, mm, ss, tzoffset(&t));
}

static void compose_generate_msgid(gchar *buf, gint len)
{
	struct tm *lt;
	time_t t;
	gchar *addr;

	t = time(NULL);
	lt = localtime(&t);

	if (cur_account && cur_account->address && *cur_account->address) {
		if (strchr(cur_account->address, '@'))
			addr = g_strdup(cur_account->address);
		else
			addr = g_strconcat(cur_account->address, "@",
					   get_domain_name(), NULL);
	} else
		addr = g_strconcat(g_get_user_name(), "@", get_domain_name(),
				   NULL);

	g_snprintf(buf, len, "%04d%02d%02d%02d%02d%02d.%08x.%s",
		   lt->tm_year + 1900, lt->tm_mon + 1,
		   lt->tm_mday, lt->tm_hour,
		   lt->tm_min, lt->tm_sec,
		   (guint)random(), addr);

	debug_print(_("generated Message-ID: %s\n"), buf);

	g_free(addr);
}

#define MAX_ENCLEN	75
#define MAX_LINELEN	76
#define JIS_SEQLEN	3

#define B64LEN(len)	((len) / 3 * 4 + ((len) % 3 ? 4 : 0))

static void compose_encode_header(gchar *dest, gint len, const gchar *src,
				  gint header_len)
{
	wchar_t *wsrc;
	wchar_t *wsrcp;
	gchar *destp;
	size_t line_len, mimehdr_len, mimehdr_begin_len;
	gchar *mimehdr_init = "=?";
	gchar *mimehdr_end = "?=";
	gchar *mimehdr_charset = OUT_CHARSET;
	gchar *mimehdr_enctype = "?B?";

	//g_print("src = %s\n", src);

	/* convert to wide-character string */
	wsrcp = wsrc = strdup_mbstowcs(src);

	mimehdr_len = strlen(mimehdr_init) + strlen(mimehdr_end) +
		      strlen(mimehdr_charset) + strlen(mimehdr_enctype);
	mimehdr_begin_len = strlen(mimehdr_init) +
			    strlen(mimehdr_charset) + strlen(mimehdr_enctype);
	//line_len = 1;
	line_len = header_len;
	destp = dest;
	*dest = '\0';

	while (*wsrcp) {
		wchar_t *wp, *wtmp, *wtmpp;
		gint nspc = 0;

		/* irresponsible buffer overrun check */
		if ((len - (destp - dest)) < (MAX_LINELEN + 1) * 2) break;

		/* encode string including space
		   if multi-byte string follows */
		if (is_next_mbs(wsrcp)) {
			wp = wsrcp;
			while ((wp = find_wspace(wp)) != NULL)
				if (!is_next_mbs(wp)) break;
		} else
			wp = find_wspace(wsrcp);

		if (wp != NULL) {
			wtmp = wcsndup(wsrcp, wp - wsrcp);
			wsrcp = wp + 1;
			while (iswspace(*(wsrcp + nspc))) nspc++;
		} else {
			wtmp = wcsdup(wsrcp);
			wsrcp += wcslen(wsrcp);
		}

		wtmpp = wtmp;

		do {
			gint prev_mbl = 1, tlen = 0, mb_seqlen = 0;
			gchar *tmp;
			register gchar *tmpp;

			tmpp = tmp = g_malloc(wcslen(wtmpp) * MB_CUR_MAX + 1);
			*tmp = '\0';

			while (*wtmpp != (wchar_t)0) {
				gint mbl;

				mbl = wctomb(tmpp, *wtmpp);
				if (mbl == -1) {
					g_warning("invalid wide character\n");
					continue;
				}

				/* length of KI + KO */
				if (prev_mbl == 1 && mbl == 2)
					mb_seqlen += JIS_SEQLEN * 2;

				if (mb_seqlen) {
					gint dlen = mimehdr_len +
						B64LEN(tlen + mb_seqlen + mbl);

					if ((line_len + dlen +
					     (*(wtmpp + 1) ? 0 : nspc) +
					     (line_len > 1 ? 1 : 0))
					    > MAX_LINELEN) {
						if (tlen == 0) {
							*destp++ = '\n';
							*destp++ = ' ';
							line_len = 1;
							mb_seqlen = 0;
							continue;
						} else {
							*tmpp = '\0';
							break;
						}
					}
				} else {
					if ((line_len + tlen + mbl +
					     (*(wtmpp + 1) ? 0 : nspc) +
					     (line_len > 1 ? 1 : 0))
					    > MAX_LINELEN) {
						if (1 + tlen + mbl +
						    (*(wtmpp + 1) ? 0 : nspc)
						    >= MAX_LINELEN) {
							*tmpp = '\0';
							break;
						}
						*destp++ = '\n';
						*destp++ = ' ';
						line_len = 1;
						continue;
					}
				}

				tmpp += mbl;
				*tmpp = '\0';

				tlen += mbl;
				prev_mbl = mbl;

				wtmpp++;
			}
			//g_print("tmp = %s, tlen = %d, mb_seqlen = %d\n",
			//	tmp, tlen, mb_seqlen);

			if (tlen == 0) {
				g_free(tmp);
				continue;
			}

			if (line_len > 1 && destp > dest) {
				*destp++ = ' ';
				*destp = '\0';
				line_len++;
			}

			if (mb_seqlen) {
				gchar *tmp_jis;

				tmp_jis = g_new(gchar, tlen + mb_seqlen + 1);
				conv_euctojis(tmp_jis,
					      tlen + mb_seqlen + 1, tmp);
				g_snprintf(destp, len - strlen(dest), "%s%s%s",
					   mimehdr_init, mimehdr_charset,
					   mimehdr_enctype);
				destp += mimehdr_begin_len;
				line_len += mimehdr_begin_len;

				to64frombits(destp, tmp_jis, strlen(tmp_jis));
				line_len += strlen(destp);
				destp += strlen(destp);

				strcpy(destp, mimehdr_end);
				destp += strlen(mimehdr_end);
				line_len += strlen(mimehdr_end);

				g_free(tmp_jis);
			} else {
				strcpy(destp, tmp);
				line_len += strlen(destp);
				destp += strlen(destp);
			}

			g_free(tmp);
			//g_print("line_len = %d\n\n", line_len);
		} while (*wtmpp != (wchar_t)0);

		while (iswspace(*wsrcp)) {
			gint mbl;

			mbl = wctomb(destp, *wsrcp++);
			if (mbl != -1) {
				destp += mbl;
				line_len += mbl;
			}
		}
		*destp = '\0';

		g_free(wtmp);
	}

	g_free(wsrc);

	//g_print("dest = %s\n", dest);
}

static GSList *compose_to_list_append(GSList *to_list, const gchar *str)
{
	gchar *work;
	gchar *workp;
	gchar *tmp, *p;

	work = g_strdup(str);
	workp = work;

	eliminate_quote(work, '"');
	eliminate_parenthesis(work, '(', ')');

	while (*workp) {
		p = strchr(workp, ',');
		if (p) {
			tmp = g_strndup(workp, p - workp);
			workp = p + 1;
		} else {
			tmp = g_strdup(workp);
			workp += strlen(workp);
		}

		if ((p = strrchr(tmp, '<')))
			extract_parenthesis(p, '<', '>');
		else
			p = tmp;

		g_strstrip(p);
		if (*p)
			to_list = g_slist_append(to_list, g_strdup(p));
		g_free(tmp);
	}

	g_free(work);
	return to_list;
}

static GSList *compose_to_list_remove_all(GSList *to_list)
{
	if (to_list != NULL) {
		GSList *cur;

		for (cur = to_list; cur != NULL; cur = cur->next)
			g_free(cur->data);
		slist_remove_all(to_list);
	}

	return NULL;
}

static Compose *compose_create(void)
{
	Compose   *compose;
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *menubar;
	GtkWidget *handlebox;
	GtkWidget *vbox2;
	GtkWidget *table_vbox;
	GtkWidget *to_entry;
	GtkWidget *subject_entry;
	GtkWidget *cc_entry;
	GtkWidget *bcc_entry;
	GtkWidget *bcc_hbox;
	GtkWidget *scrolledwin;
	GtkWidget *text;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *hbox;
	guint      n_menu_entries;
	GtkStyle  *style;
	GtkItemFactory *ifactory;

	debug_print(_("Creating compose window...\n"));
	compose = g_new0(Compose, 1);

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), _("Compose message"));
	gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
	gtk_widget_set_usize(window, -1, DEFAULT_COMPOSE_HEIGHT);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
			   GTK_SIGNAL_FUNC(gtk_false), NULL);
	gtk_signal_connect(GTK_OBJECT(window), "destroy",
			   GTK_SIGNAL_FUNC(gtk_false), NULL);
	gtk_widget_realize(window);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	n_menu_entries = sizeof(compose_entries) / sizeof(compose_entries[0]);
	menubar = menubar_create(window, compose_entries,
				 n_menu_entries, "<Compose>", compose);
	gtk_box_pack_start(GTK_BOX(vbox), menubar, FALSE, TRUE, 0);

	handlebox = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(vbox), handlebox, FALSE, FALSE, 0);

	compose_toolbar_create(compose, handlebox);

	vbox2 = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), vbox2, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox2), BORDER_WIDTH);

	table_vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox2), table_vbox, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(table_vbox),
				       BORDER_WIDTH * 2);

	table = gtk_table_new(4, 2, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 4);
	gtk_box_pack_start(GTK_BOX(table_vbox), table, FALSE, TRUE, 0);

	/* header labels */
	hbox = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new(prefs_common.trans_hdr ? _("Subject:")
			      : "Subject:");
	gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 0, 1,
			 GTK_FILL, 0, 2, 0);
	hbox = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new(prefs_common.trans_hdr ? _("To:")
			      : "To:");
	gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 1, 2,
			 GTK_FILL, 0, 2, 0);
	hbox = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new(prefs_common.trans_hdr ? _("Cc:")
			      : "Cc:");
	gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox, 0, 1, 2, 3,
			 GTK_FILL, 0, 2, 0);

	bcc_hbox = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new(prefs_common.trans_hdr ? _("Bcc:")
			      : "Bcc:");
	gtk_box_pack_end(GTK_BOX(bcc_hbox), label, FALSE, FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), bcc_hbox, 0, 1, 3, 4,
			 GTK_FILL, 0, 2, 0);

	/* header entries */
	subject_entry = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), subject_entry, 1, 2, 0, 1);
	to_entry = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), to_entry, 1, 2, 1, 2);
	cc_entry = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), cc_entry, 1, 2, 2, 3);

	bcc_entry = gtk_entry_new();
	gtk_table_attach_defaults(GTK_TABLE(table), bcc_entry, 1, 2, 3, 4);

	gtk_signal_connect(GTK_OBJECT(subject_entry), "activate",
			   GTK_SIGNAL_FUNC(subject_activated), compose);
	gtk_signal_connect(GTK_OBJECT(to_entry), "activate",
			   GTK_SIGNAL_FUNC(to_activated), compose);
	gtk_signal_connect(GTK_OBJECT(cc_entry), "activate",
			   GTK_SIGNAL_FUNC(cc_activated), compose);
	gtk_signal_connect(GTK_OBJECT(bcc_entry), "activate",
			   GTK_SIGNAL_FUNC(bcc_activated), compose);

	scrolledwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwin),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(vbox2), scrolledwin, TRUE, TRUE, 0);
	gtk_widget_set_usize(scrolledwin, DEFAULT_MAINVIEW_WIDTH, -1);

	text = gtk_text_new(gtk_scrolled_window_get_hadjustment
			    (GTK_SCROLLED_WINDOW(scrolledwin)),
			    gtk_scrolled_window_get_vadjustment
			    (GTK_SCROLLED_WINDOW(scrolledwin)));
	gtk_text_set_editable(GTK_TEXT(text), TRUE);
	gtk_container_add(GTK_CONTAINER(scrolledwin), text);

	gtk_widget_show_all(vbox);
	gtk_widget_hide(bcc_hbox);
	gtk_widget_hide(bcc_entry);

	if (prefs_common.messagefont) {
		GdkFont *font;

		font = gdk_fontset_load(prefs_common.messagefont);
		if (font) {
			style = gtk_style_copy(gtk_widget_get_style(text));
			style->font = font;
			gtk_widget_set_style(text, style);
		}
	}

	ifactory = gtk_item_factory_from_widget(menubar);
	menu_set_sensitive(ifactory, "/File/Attach file", FALSE);
	menu_set_sensitive(ifactory, "/Edit/Undo",	  FALSE);
	menu_set_sensitive(ifactory, "/Edit/Redo",	  FALSE);
	menu_set_sensitive(ifactory, "/Message/Forward",  FALSE);
	menu_set_sensitive(ifactory, "/Message/Draft",	  FALSE);

	switch (prefs_common.toolbar_style) {
	case TOOLBAR_NONE:
		gtk_widget_hide(handlebox);
		break;
	case TOOLBAR_ICON:
		gtk_toolbar_set_style(GTK_TOOLBAR(compose->toolbar),
				      GTK_TOOLBAR_ICONS);
		break;
	case TOOLBAR_TEXT:
		gtk_toolbar_set_style(GTK_TOOLBAR(compose->toolbar),
				      GTK_TOOLBAR_TEXT);
		break;
	case TOOLBAR_BOTH:
		gtk_toolbar_set_style(GTK_TOOLBAR(compose->toolbar),
				      GTK_TOOLBAR_BOTH);
		break;
	}

	gtk_widget_show(window);

	compose->window        = window;
	compose->vbox	       = vbox;
	compose->menubar       = menubar;
	compose->handlebox     = handlebox;

	compose->table_vbox    = table_vbox;
	compose->table	       = table;
	compose->to_entry      = to_entry;
	compose->subject_entry = subject_entry;
	compose->cc_entry      = cc_entry;
	compose->bcc_hbox      = bcc_hbox;
	compose->bcc_entry     = bcc_entry;

	compose->scrolledwin   = scrolledwin;
	compose->text	       = text;

	compose->cc	    = NULL;
	compose->replyto    = NULL;
	compose->inreplyto  = NULL;
	compose->references = NULL;
	compose->to_list    = NULL;

	compose->use_bcc = FALSE;

	return compose;
}

#include "pixmaps/stock_mail_send.xpm"
#include "pixmaps/stock_mail.xpm"

#define CREATE_TOOLBAR_ICON(xpm_d) \
{ \
	icon = gdk_pixmap_create_from_xpm_d(container->window, &mask, \
					    &container->style->white, \
					    xpm_d); \
	icon_wid = gtk_pixmap_new(icon, mask); \
}

static void compose_toolbar_create(Compose *compose, GtkWidget *container)
{
	GtkWidget *toolbar;
	GdkPixmap *icon;
	GdkBitmap *mask;
	GtkWidget *icon_wid;
	GtkWidget *send_btn;
	GtkWidget *insert_btn;
	GtkWidget *sig_btn;
	GtkWidget *addrbook_btn;

	toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,
				  GTK_TOOLBAR_BOTH);
	gtk_container_add(GTK_CONTAINER(container), toolbar);
	gtk_container_set_border_width(GTK_CONTAINER(container), 2);
	gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar), GTK_RELIEF_NONE);
	gtk_toolbar_set_space_style(GTK_TOOLBAR(toolbar),
				    GTK_TOOLBAR_SPACE_LINE);

	CREATE_TOOLBAR_ICON(stock_mail_send_xpm);
	send_btn = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),
					   _("Send"),
					   _("Send message"),
					   "Send",
					   icon_wid, toolbar_send_cb, compose);

	gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

	CREATE_TOOLBAR_ICON(stock_mail_xpm);
	insert_btn = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),
					     _("Insert"),
					     _("Insert file"),
					     "Insert",
					     icon_wid, toolbar_insert_cb,
					     compose);

	CREATE_TOOLBAR_ICON(stock_mail_xpm);
	sig_btn = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),
					  _("Signature"),
					  _("Insert signature"),
					  "Signature",
					  icon_wid, toolbar_sig_cb, compose);

	gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

	CREATE_TOOLBAR_ICON(stock_mail_xpm);
	addrbook_btn = gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),
					       _("Address"),
					       _("Address book"),
					       "Address",
					       icon_wid, NULL, compose);
	gtk_widget_set_sensitive(addrbook_btn, FALSE);

	compose->toolbar      = toolbar;
	compose->send_btn     = send_btn;
	compose->insert_btn   = insert_btn;
	compose->sig_btn      = sig_btn;
	compose->addrbook_btn = addrbook_btn;

	gtk_widget_show_all(toolbar);
}

static void compose_destroy(Compose *compose)
{
	compose_to_list_remove_all(compose->to_list);

	g_free(compose->replyto);
	g_free(compose->cc);
	g_free(compose->inreplyto);
	g_free(compose->references);

	gtk_widget_destroy(compose->window);

	g_free(compose);
}

/* callback functions */

static void toolbar_send_cb(GtkWidget *widget, gpointer data)
{
	compose_send_cb(data, 0, NULL);
}

static void toolbar_insert_cb(GtkWidget *widget, gpointer data)
{
	compose_insert_file_cb(data, 0, NULL);
}

static void toolbar_sig_cb(GtkWidget *widget, gpointer data)
{
	Compose *compose = (Compose *)data;

	compose_insert_sig(compose);
}

static void compose_send_cb(gpointer data, guint action, GtkWidget *widget)
{
	Compose *compose = (Compose *)data;
	gint val;

	val = compose_send(compose);
	if (val < 0 && !prefs_common.queue_msg) return;
	if (val > 0) return;

	compose_destroy(compose);
}

static void compose_insert_file_cb(gpointer data, guint action,
				   GtkWidget *widget)
{
	Compose *compose = (Compose *)data;
	gchar *file;

	file = filesel_select_file(_("Select file"));

	if (file)
		compose_insert_file(compose, file);
}

static void compose_close_cb(gpointer data, guint action, GtkWidget *widget)
{
	Compose *compose = (Compose *)data;

	compose_destroy(compose);
}

static void compose_cut_cb(Compose *compose)
{
	gtk_editable_cut_clipboard(GTK_EDITABLE(compose->text));
}

static void compose_copy_cb(Compose *compose)
{
	gtk_editable_copy_clipboard(GTK_EDITABLE(compose->text));
}

static void compose_paste_cb(Compose *compose)
{
	gtk_editable_paste_clipboard(GTK_EDITABLE(compose->text));
}

static void compose_allsel_cb(Compose *compose)
{
	gtk_editable_select_region(GTK_EDITABLE(compose->text), 0, -1);
}

static void compose_toggle_bcc_cb(gpointer data, guint action,
				  GtkWidget *widget)
{
	Compose *compose = (Compose *)data;

	if (GTK_CHECK_MENU_ITEM(widget)->active) {
		gtk_widget_show(compose->bcc_hbox);
		gtk_widget_show(compose->bcc_entry);
		compose->use_bcc = TRUE;
	} else {
		gtk_widget_hide(compose->bcc_hbox);
		gtk_widget_hide(compose->bcc_entry);
		gtk_widget_queue_resize(compose->table_vbox);
		compose->use_bcc = FALSE;
	}
}

static void subject_activated(GtkWidget *widget, Compose *compose)
{
	gtk_widget_grab_focus(compose->to_entry);
}

static void to_activated(GtkWidget *widget, Compose *compose)
{
	gtk_widget_grab_focus(compose->cc_entry);
}

static void cc_activated(GtkWidget *widget, Compose *compose)
{
	if (compose->use_bcc)
		gtk_widget_grab_focus(compose->bcc_entry);
	else
		gtk_widget_grab_focus(compose->text);
}

static void bcc_activated(GtkWidget *widget, Compose *compose)
{
	gtk_widget_grab_focus(compose->text);
}
