/* filter.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <string.h>

#include "intl.h"
#include "procheader.h"
#include "filter.h"
#include "utils.h"

gchar *filter_get_dest_folder(GSList *fltlist, const gchar *file)
{
	static gchar *dest_folder = NULL;
	GSList *hlist, *cur;
	Filter *filter;

	g_return_val_if_fail(fltlist != NULL, NULL);
	g_return_val_if_fail(file != NULL, NULL);

	g_free(dest_folder);
	dest_folder = NULL;

	hlist = procheader_get_header_list(file);
	if (!hlist) return NULL;

	for (cur = fltlist; cur != NULL; cur = cur->next) {
		filter = (Filter *)cur->data;
		if (filter_match_condition(filter, hlist)) {
			dest_folder = g_strdup(filter->dest);
			break;
		}
	}

	procheader_header_list_destroy(hlist);

	return dest_folder;
}

gboolean filter_match_condition(Filter *filter, GSList *hlist)
{
	Header *header;
	gint   (*StrCompare) (const gchar *s1, const gchar *s2);
	gchar *(*StrFind)    (const gchar *hs, const gchar *nd);

	if (filter->case_sens) {
		StrCompare = strcmp;
		StrFind    = strstr;
	} else {
		StrCompare = strcasecmp;
		StrFind    = strcasestr;
	}

	if (filter->cond == FLT_AND) {
		gboolean match1 = FALSE, match2 = FALSE;

		/* ignore second condition if not set */
		if (!filter->name2) match2 = TRUE;

		for (; hlist != NULL; hlist = hlist->next) {
			header = hlist->data;

			if (!match1 &&
			    !StrCompare(header->name, filter->name1)) {
				if (StrFind(header->body, filter->body1)
				    != NULL)
					match1 = TRUE;
			}
			if (!match2 &&
			    !StrCompare(header->name, filter->name2)) {
				if (StrFind(header->body, filter->body2)
				    != NULL)
					match2 = TRUE;
			}

			if (match1 && match2) return TRUE;
		}
	} else if (filter->cond == FLT_OR) {
		for (; hlist != NULL; hlist = hlist->next) {
			header = hlist->data;

			if (!StrCompare(header->name, filter->name1))
				if (StrFind(header->body, filter->body1)
				    != NULL)
					return TRUE;
			if (filter->name2 &&
			    !StrCompare(header->name, filter->name2))
				if (StrFind(header->body, filter->body2)
				    != NULL)
					return TRUE;
		}
	}

	return FALSE;
}

gchar *filter_get_str(Filter *filter)
{
	gchar *str;

	str = g_strdup_printf
		("%s\t%s\t%c\t%s\t%s\t%s",
		 filter->name1, filter->body1 ? filter->body1 : "",
		 filter->name2 ? (filter->cond == FLT_AND ? '&' : '|') : ' ',
		 filter->name2 ? filter->name2 : "",
		 filter->body2 ? filter->body2 : "",
		 filter->dest);

	return str;
}

#define PARSE_ONE_PARAM(p, srcp) \
{ \
	p = strchr(srcp, '\t'); \
	if (!p) { \
		g_free(tmp); \
		return NULL; \
	} else \
		*p++ = '\0'; \
}

Filter *filter_read_str(const gchar *str)
{
	Filter *filter;
	gchar *tmp;
	gchar *name1, *body1, *op, *name2, *body2, *dest;

	tmp = g_strdup(str);

	name1 = tmp;
	PARSE_ONE_PARAM(body1, name1);
	PARSE_ONE_PARAM(op, body1);
	PARSE_ONE_PARAM(name2, op);
	PARSE_ONE_PARAM(body2, name2);
	PARSE_ONE_PARAM(dest, body2);

	filter = g_new(Filter, 1);
	filter->name1 = *name1 ? g_strdup(name1) : NULL;
	filter->body1 = *body1 ? g_strdup(body1) : NULL;
	filter->name2 = *name2 ? g_strdup(name2) : NULL;
	filter->body2 = *body2 ? g_strdup(body2) : NULL;
	filter->cond = (*op == '|') ? FLT_OR : FLT_AND;
	filter->case_sens = FALSE;
	filter->dest = *dest ? g_strdup(dest) : NULL;

	return filter;
}

void filter_free(Filter *filter)
{
	if (!filter) return;

	g_free(filter->name1);
	g_free(filter->body1);

	g_free(filter->name2);
	g_free(filter->body2);

	g_free(filter->dest);

	g_free(filter);
}
