/* foldersel.c */

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkctree.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkhbbox.h>
#include <gtk/gtksignal.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>

#include "main.h"
#include "utils.h"
#include "gtkutils.h"
#include "foldersel.h"
#include "alertpanel.h"
#include "manage_window.h"

#include "pixmaps/dir-close.xpm"
#include "pixmaps/dir-open.xpm"

static GdkPixmap *folderxpm;
static GdkBitmap *folderxpmmask;
static GdkPixmap *folderopenxpm;
static GdkBitmap *folderopenxpmmask;

static GtkWidget *window;
static GtkWidget *ctree;
static GtkWidget *entry;
static GtkWidget *ok_button;
static GtkWidget *cancel_button;
static gchar *folder;

static gboolean cancelled;

static void foldersel_create(void);
static void foldersel_init(void);
static void foldersel_free_str(GtkCTree *ctree, GtkCTreeNode *node,
			       gpointer data);
static void foldersel_set_tree(void);
static void foldersel_selected(GtkCList *clist, gint row, gint column,
			       GdkEvent *event, gpointer data);

static void foldersel_ok(GtkButton *button, gpointer data);
static void foldersel_cancel(GtkButton *button, gpointer data);
static void foldersel_activated(void);
static void key_pressed(GtkWidget *widget, GdkEventKey *event, gpointer data);

gchar *foldersel_folder_sel(const gchar *default_folder)
{
	GtkCTreeNode *node;

	if (!window) {
		foldersel_create();
		foldersel_init();
	} else
		gtk_widget_show(window);
	manage_window_set_transient(GTK_WINDOW(window));

	foldersel_set_tree();

	if (folder) {
		debug_print(_("Previously selected folder: %s\n"), folder);
		node = gtk_ctree_find_by_row_data_custom
			(GTK_CTREE(ctree), NULL, folder,
			 (GCompareFunc)path_cmp);
		if (node)
			gtk_ctree_select(GTK_CTREE(ctree), node);
	}
	gtk_widget_grab_focus(ok_button);
	gtk_widget_grab_focus(entry);

	cancelled = FALSE;

	gtk_main();

	gtk_widget_hide(window);
	gtk_entry_set_text(GTK_ENTRY(entry), "");
	gtk_ctree_pre_recursive(GTK_CTREE(ctree), NULL,
				foldersel_free_str, NULL);
	gtk_clist_clear(GTK_CLIST(ctree));

	if (!cancelled) {
		debug_print("folder: %s\n", folder);
		return folder;
	} else
		return NULL;
}

static void foldersel_create(void)
{
	GtkWidget *vbox;
	GtkWidget *scrolledwin;
	GtkWidget *confirm_area;

	window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(window), _("Select folder"));
	gtk_widget_set_usize(window, 300, 400);
	gtk_container_set_border_width(GTK_CONTAINER(window), BORDER_WIDTH);
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
	gtk_window_set_modal(GTK_WINDOW(window), TRUE);
	gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, TRUE);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
			   GTK_SIGNAL_FUNC(foldersel_cancel), NULL);
	gtk_signal_connect(GTK_OBJECT(window), "key_press_event",
			   GTK_SIGNAL_FUNC(key_pressed), NULL);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	scrolledwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwin),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(vbox), scrolledwin, TRUE, TRUE, 0);

	ctree = gtk_ctree_new(1, 0);
	gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(scrolledwin),
					    GTK_CLIST(ctree)->vadjustment);
	gtk_container_add(GTK_CONTAINER(scrolledwin), ctree);
	gtk_clist_set_selection_mode(GTK_CLIST(ctree), GTK_SELECTION_BROWSE);
	gtk_ctree_set_line_style(GTK_CTREE(ctree), GTK_CTREE_LINES_DOTTED);
	gtk_ctree_set_indent(GTK_CTREE(ctree), CTREE_INDENT);
	GTK_WIDGET_UNSET_FLAGS(GTK_CLIST(ctree)->column[0].button,
			       GTK_CAN_FOCUS);
	//gtk_signal_connect(GTK_OBJECT(ctree), "tree_select_row",
	//		   GTK_SIGNAL_FUNC(foldersel_selected), NULL);
	gtk_signal_connect(GTK_OBJECT(ctree), "select_row",
			   GTK_SIGNAL_FUNC(foldersel_selected), NULL);

	entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(entry), "activate",
			   GTK_SIGNAL_FUNC(foldersel_activated), NULL);

	gtkut_button_set_create(&confirm_area,
				&ok_button,	_("OK"),
				&cancel_button,	_("Cancel"),
				NULL, NULL);

	gtk_box_pack_end(GTK_BOX(vbox), confirm_area, FALSE, FALSE, 0);
	gtk_widget_grab_default(ok_button);

	gtk_signal_connect(GTK_OBJECT(ok_button), "clicked",
			   GTK_SIGNAL_FUNC(foldersel_ok), NULL);
	gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked",
			   GTK_SIGNAL_FUNC(foldersel_cancel), NULL);

	gtk_widget_show_all(window);
}

static void foldersel_init(void)
{
	PIXMAP_CREATE(ctree, folderxpm, folderxpmmask, DIRECTORY_CLOSE_XPM);
	PIXMAP_CREATE(ctree, folderopenxpm, folderopenxpmmask,
		      DIRECTORY_OPEN_XPM);
}

static void foldersel_free_str(GtkCTree *ctree, GtkCTreeNode *node,
			       gpointer data)
{
	gchar *ptr = gtk_ctree_node_get_row_data(ctree, node);

	g_free(ptr);
}

static void foldersel_set_tree(void)
{
	FILE *fp;
	gchar dir[MAXPATHLEN], parent[MAXPATHLEN];
	gchar *p;
	gchar *fcache;
	gchar *text[1] = {"Mailbox"};
	GtkCTreeNode *tnode, *node;

	CHDIR_RETURN_IF_FAIL(maildir);

	fcache = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, RC_DIR,
			     G_DIR_SEPARATOR_S, MAIL_FOLDER_LIST, NULL);
	if ((fp = fopen(fcache, "r")) == NULL) {
		fprintf(stderr, "%s: ", fcache);
		perror("fopen");
		g_free(fcache);
		return;
	}
	g_free(fcache);

	gtk_clist_freeze(GTK_CLIST(ctree));

	tnode = gtk_ctree_insert_node(GTK_CTREE(ctree), NULL, NULL,
				      text, FOLDER_SPACING,
				      folderxpm, folderxpmmask,
				      folderopenxpm, folderopenxpmmask,
				      FALSE, TRUE);

	while (fgets(dir, sizeof(dir), fp) != NULL) {
		dir[strlen(dir) - 1] = '\0';
		/* get parent directory */
		if ((p = strrchr(dir, '/')) != NULL) {
			strncpy(parent, dir, (size_t)(p - dir));
			parent[p - dir] = '\0';
			node = gtk_ctree_find_by_row_data_custom
				(GTK_CTREE(ctree), tnode, parent,
				 (GCompareFunc)strcmp2);
			gtk_ctree_expand(GTK_CTREE(ctree), node);
		} else
			node = tnode;
		text[0] = g_basename(dir);
		node = gtk_ctree_insert_node
			(GTK_CTREE(ctree), node, NULL, text, FOLDER_SPACING,
			 folderxpm, folderxpmmask,
			 folderopenxpm, folderopenxpmmask, FALSE, FALSE);
		gtk_ctree_node_set_row_data(GTK_CTREE(ctree), node,
					    g_strdup(dir));
		fgets(dir, sizeof(dir), fp);
	}
	fclose(fp);

	gtk_ctree_sort_recursive(GTK_CTREE(ctree), tnode);

	gtk_clist_thaw(GTK_CLIST(ctree));
}

static void foldersel_selected(GtkCList *clist, gint row, gint column,
			       GdkEvent *event, gpointer data)
{
	gchar *dir;
	GdkEventButton *ev = (GdkEventButton *)event;

	dir = gtk_clist_get_row_data(clist, row);
	if (dir) gtk_entry_set_text(GTK_ENTRY(entry), dir);

	if (ev && GDK_2BUTTON_PRESS == ev->type)
		gtk_button_clicked(GTK_BUTTON(ok_button));
}

static void foldersel_ok(GtkButton *button, gpointer data)
{
	gchar *dir;
	struct stat s;

	dir = gtk_entry_get_text(GTK_ENTRY(entry));
	if (!dir || *dir == '\0') return;

	if (stat(dir, &s) < 0) {
		g_print("stat: ");
		perror(dir);
		if (alertpanel(_("New directory"),
			       _("The directory not found. Create it?"),
			       _("Yes"), _("No"), NULL) == G_ALERTDEFAULT) {
			if (mkdir(dir, S_IRWXU) < 0) {
				perror("mkdir");
				alertpanel(_("Error"),
					   _("Can't create directory."),
					   _("OK"), NULL, NULL);
				return;
			}
			if (chmod(dir, S_IRWXU) < 0) {
				fprintf(stderr, "%s: ", dir);
				perror("chmod");
			}
		} else
			return;
	} else {
		if (!S_ISDIR(s.st_mode)) {
			alertpanel(_("Error"),
				   _("Selected name isn't a directory."),
				   _("OK"), NULL, NULL);
			return;
		}
	}

	g_free(folder);
	folder = g_strdup(dir);

	gtk_main_quit();
}

static void foldersel_cancel(GtkButton *button, gpointer data)
{
	cancelled = TRUE;
	gtk_main_quit();
}

static void foldersel_activated(void)
{
	gtk_button_clicked(GTK_BUTTON(ok_button));
}

static void key_pressed(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	if (event && event->keyval == GDK_Escape) {
		cancelled = TRUE;
		gtk_main_quit();
	}
}
