/* folderview.c */

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkctree.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkstyle.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkstatusbar.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkitemfactory.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "main.h"
#include "mainwindow.h"
#include "folderview.h"
#include "summaryview.h"
#include "inputdialog.h"
#include "alertpanel.h"
#include "menu.h"
#include "procmsg.h"
#include "utils.h"
#include "gtkutils.h"

#include "pixmaps/inbox.xpm"
#include "pixmaps/outbox.xpm"
#include "pixmaps/dir-close.xpm"
#include "pixmaps/dir-open.xpm"
#include "pixmaps/trash.xpm"

typedef enum
{
	COL_FOLDER	= 0,
	COL_NEW		= 1,
	COL_UNREAD	= 2,
	COL_TOTAL	= 3
} FolderColumnPos;

#define N_FOLDER_COLS		4
#define COL_FOLDER_WIDTH	150
#define COL_NUM_WIDTH		32

#define IS_SPECIAL_FOLDER(folder) \
	(!strcmp(folder, INBOX_DIR)      || \
	 !strcmp(folder, OUTBOX_DIR)     || \
	 !strcmp(folder, QUEUE_DIR)      || \
	 !strcmp(folder, DRAFT_DIR)      || \
	 !strcmp(folder, TRASH_DIR))

#define STATUSBAR_PUSH(mainwin, str) \
{ \
	gtk_statusbar_push(GTK_STATUSBAR(mainwin->statusbar), \
			   mainwin->folderview_cid, str); \
}

#define STATUSBAR_POP(mainwin) \
{ \
	gtk_statusbar_pop(GTK_STATUSBAR(mainwin->statusbar), \
			  mainwin->folderview_cid); \
}

static GdkFont *normalfont;
static GdkFont *boldfont;

static GdkPixmap *inboxxpm;
static GdkBitmap *inboxxpmmask;
static GdkPixmap *outboxxpm;
static GdkBitmap *outboxxpmmask;
static GdkPixmap *folderxpm;
static GdkBitmap *folderxpmmask;
static GdkPixmap *folderopenxpm;
static GdkBitmap *folderopenxpmmask;
static GdkPixmap *trashxpm;
static GdkBitmap *trashxpmmask;

static void folderview_set_mailbox_folder(FolderView	*folderview);
static void folderview_set_news_folder	 (FolderView	*folderview);
static GtkCTreeNode *set_special_folder	 (FolderView	*folderview,
					  GtkCTreeNode	*prevnode,
					  const gchar	*folder,
					  const gchar	*newname,
					  GdkPixmap	*xpm,
					  GdkBitmap	*mask);

static void folderview_set_special_folders	(FolderView	*folderview);
static gint folderview_read_list		(FolderView	*folderview,
						 FolderType	 type);
static void folderview_write_cache_func		(GtkCTree	*ctree,
						 GtkCTreeNode	*node,
						 gpointer	 data);
static void folderview_scan_mailbox		(FolderView	*folderview,
						 GtkCTreeNode	*node,
						 const gchar	*dir);

/* callback functions */
static void folderview_button_pressed	(GtkWidget	*ctree,
					 GdkEventButton	*event,
					 FolderView	*folderview);
static void folderview_button_released	(GtkWidget	*ctree,
					 GdkEventButton	*event,
					 FolderView	*folderview);
static void folderview_key_pressed	(GtkWidget	*widget,
					 GdkEventKey	*event,
					 FolderView	*folderview);
static void folderview_selected		(GtkCTree	*ctree,
					 GtkCTreeNode	*row,
					 gint		 column,
					 FolderView	*folderview);
static void folderview_popup_close	(GtkMenuShell	*menu_shell,
					 FolderView	*folderview);

static void folderview_new_folder_cb	(FolderView	*folderview,
					 guint		 action,
					 GtkWidget	*widget);
static void folderview_rename_folder_cb	(FolderView	*folderview,
					 guint		 action,
					 GtkWidget	*widget);
static void folderview_delete_folder_cb	(FolderView	*folderview,
					 guint		 action,
					 GtkWidget	*widget);
static void folderview_new_group_cb	(FolderView	*folderview,
					 guint		 action,
					 GtkWidget	*widget);
static void folderview_rm_group_cb	(FolderView	*folderview,
					 guint		 action,
					 GtkWidget	*widget);
static void folderview_add_server_cb	(FolderView	*folderview,
					 guint		 action,
					 GtkWidget	*widget);
static void folderview_rm_server_cb	(FolderView	*folderview,
					 guint		 action,
					 GtkWidget	*widget);

static void folderview_free_data_func	(GtkCTree	*ctree,
					 GtkCTreeNode	*node,
					 gpointer	 data);
static gint folderview_compare_path	(gconstpointer	 a,
					 gconstpointer	 b);

static GtkItemFactoryEntry folderview_mail_popup_entries[] =
{
	{N_("/Create _new folder"),	NULL, folderview_new_folder_cb,    0, NULL},
	{N_("/_Rename folder"),		NULL, folderview_rename_folder_cb, 0, NULL},
	{N_("/_Delete folder"),		NULL, folderview_delete_folder_cb, 0, NULL}
};

static GtkItemFactoryEntry folderview_news_popup_entries[] =
{
	{N_("/_Subscribe to newsgroup"), NULL, folderview_new_group_cb,  0, NULL},
	{N_("/_Remove newsgroup"),	 NULL, folderview_rm_group_cb,   0, NULL},
	{N_("/---"),			 NULL, NULL, 0, "<Separator>"},
	{N_("/_Add news server"),	 NULL, folderview_add_server_cb, 0, NULL},
	{N_("/Remove _news server"),	 NULL, folderview_rm_server_cb,  0, NULL}
};

FolderView *folderview_create(void)
{
	FolderView *folderview;
	GtkWidget *scrolledwin;
	GtkWidget *ctree;
	gchar *titles[N_FOLDER_COLS] = {_("Folder"), _("New"),
					_("Unread"), _("#")};
	GtkWidget *mail_popup;
	GtkWidget *news_popup;
	GtkItemFactory *mail_factory;
	GtkItemFactory *news_factory;
	gint n_entries;
	gint i;

	debug_print(_("Creating folder view...\n"));
	folderview = g_new0(FolderView, 1);

	scrolledwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_ALWAYS);
	gtk_widget_set_usize(scrolledwin,
			     COL_FOLDER_WIDTH + COL_NUM_WIDTH - 3, -1);

	ctree = gtk_ctree_new_with_titles(N_FOLDER_COLS, COL_FOLDER, titles);
	gtk_container_add(GTK_CONTAINER(scrolledwin), ctree);
	gtk_clist_set_selection_mode(GTK_CLIST(ctree), GTK_SELECTION_BROWSE);
	gtk_clist_set_column_justification(GTK_CLIST(ctree), COL_NEW,
					   GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(GTK_CLIST(ctree), COL_UNREAD,
					   GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(GTK_CLIST(ctree), COL_TOTAL,
					   GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_FOLDER,
				   COL_FOLDER_WIDTH);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_NEW,
				   COL_NUM_WIDTH);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_UNREAD,	
				   COL_NUM_WIDTH);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_TOTAL,
				   COL_NUM_WIDTH);
	gtk_ctree_set_line_style(GTK_CTREE(ctree), GTK_CTREE_LINES_DOTTED);
	gtk_ctree_set_indent(GTK_CTREE(ctree), CTREE_INDENT);
	//gtk_clist_set_reorderable(GTK_CLIST(ctree), TRUE);
	/* don't let title buttons take key focus */
	for (i = 0; i < N_FOLDER_COLS; i++)
		GTK_WIDGET_UNSET_FLAGS(GTK_CLIST(ctree)->column[i].button,
				       GTK_CAN_FOCUS);

	/* popup menu */
	n_entries = sizeof(folderview_mail_popup_entries) /
		sizeof(folderview_mail_popup_entries[0]);
	mail_popup = menu_create_items(folderview_mail_popup_entries,
				       n_entries,
				       "<MailFolder>", &mail_factory,
				       folderview);
	n_entries = sizeof(folderview_news_popup_entries) /
		sizeof(folderview_news_popup_entries[0]);
	news_popup = menu_create_items(folderview_news_popup_entries,
				       n_entries,
				       "<NewsFolder>", &news_factory,
				       folderview);

	gtk_signal_connect(GTK_OBJECT(ctree), "tree_select_row",
			   GTK_SIGNAL_FUNC(folderview_selected), folderview);
	gtk_signal_connect(GTK_OBJECT(ctree), "button_press_event",
			   GTK_SIGNAL_FUNC(folderview_button_pressed),
			   folderview);
	gtk_signal_connect(GTK_OBJECT(ctree), "button_release_event",
			   GTK_SIGNAL_FUNC(folderview_button_released),
			   folderview);
	gtk_signal_connect(GTK_OBJECT(ctree), "key_press_event",
			   GTK_SIGNAL_FUNC(folderview_key_pressed),
			   folderview);
	gtk_signal_connect(GTK_OBJECT(mail_popup), "selection_done",
			   GTK_SIGNAL_FUNC(folderview_popup_close),
			   folderview);
	gtk_signal_connect(GTK_OBJECT(news_popup), "selection_done",
			   GTK_SIGNAL_FUNC(folderview_popup_close),
			   folderview);

	folderview->scrolledwin  = scrolledwin;
	folderview->ctree        = ctree;
	folderview->mail_popup   = mail_popup;
	folderview->mail_factory = mail_factory;
	folderview->news_popup   = news_popup;
	folderview->news_factory = news_factory;

	gtk_widget_show_all(scrolledwin);

	return folderview;
}

void folderview_init(FolderView *folderview)
{
	GtkWidget *ctree = folderview->ctree;

	PIXMAP_CREATE(ctree, inboxxpm, inboxxpmmask, inbox_xpm);
	PIXMAP_CREATE(ctree, outboxxpm, outboxxpmmask, outbox_xpm);
	PIXMAP_CREATE(ctree, folderxpm, folderxpmmask, DIRECTORY_CLOSE_XPM);
	PIXMAP_CREATE(ctree, folderopenxpm, folderopenxpmmask,
		      DIRECTORY_OPEN_XPM);
	PIXMAP_CREATE(ctree, trashxpm, trashxpmmask, trash_xpm);

	if (!normalfont)
		normalfont = gdk_fontset_load(NORMAL_FONT);
	if (!boldfont) {
		boldfont = gdk_fontset_load(BOLD_FONT);
	}
}

void folderview_set(FolderView *folderview)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	MainWindow *mainwin = folderview->mainwin;

	CHDIR_RETURN_IF_FAIL(maildir);

	debug_print(_("Setting folder info...\n"));
	STATUSBAR_PUSH(mainwin, _("Setting folder info..."));

	main_window_cursor_wait(mainwin);
	gtk_clist_freeze(GTK_CLIST(ctree));

	folderview_set_mailbox_folder(folderview);
	folderview_set_news_folder(folderview);

	if (folderview_read_list(folderview, F_MHFOLDER) < 0)
		folderview_scan_mailbox(folderview, folderview->mailbox, ".");
	folderview_read_list(folderview, F_NEWSGROUP);

	gtk_ctree_sort_recursive(ctree, folderview->mailbox);
	folderview_set_special_folders(folderview);
	folderview_write_cache(folderview);

	gtk_clist_thaw(GTK_CLIST(ctree));
	main_window_cursor_normal(mainwin);
	STATUSBAR_POP(mainwin);
}

void folderview_select(FolderView *folderview, const gchar *dir)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	GtkCTreeNode *node;

	if (!dir) return;

	node = gtk_ctree_find_by_row_data_custom
		(ctree, NULL, (gpointer)dir,
		 (GCompareFunc)folderview_compare_path);
	if (node) {
		folderview->open_folder = TRUE;
		gtk_ctree_select(ctree, node);
		gtkut_ctree_set_focus_row(ctree, node);
	}
}

void folderview_unselect(FolderView *folderview)
{
	folderview->selected = folderview->opened = NULL;
}

void folderview_select_next_unread(FolderView *folderview)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	GtkCTreeNode *node;
	FolderInfo *finfo;

	if (folderview->opened)
		node = GTK_CTREE_NODE_NEXT(folderview->opened);
	else
		node = gtk_ctree_node_nth(ctree, 1);

	for (; node != NULL; node = GTK_CTREE_NODE_NEXT(node)) {
		finfo = gtk_ctree_node_get_row_data(ctree, node);
		if (finfo && (finfo->unread > 0 || finfo->new > 0)) {
			folderview->open_folder = TRUE;
			gtk_ctree_select(ctree, node);
			gtkut_ctree_set_focus_row(ctree, node);
			return;
		}
	}
}

void folderview_write_cache(FolderView *folderview)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	GtkCTreeNode *node;
	SummaryView *summaryview = folderview->summaryview;
	gchar *fcache;
	FILE *fp;

	if (summaryview && summaryview->cur_folder) {
		node = gtk_ctree_find_by_row_data_custom
			(ctree, NULL, summaryview->cur_folder, /* fixme */
			 (GCompareFunc)folderview_compare_path);
		if (node)
			folderview_update_msg_num(folderview, node,
						  summaryview->newmsgs,
						  summaryview->unread,
						  summaryview->messages);
	}

	fcache = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, RC_DIR,
			     G_DIR_SEPARATOR_S, MAIL_FOLDER_LIST, NULL);
	if ((fp = fopen(fcache, "w")) == NULL) {
		fprintf(stderr, "%s: ", fcache);
		perror("fopen");
	} else {
		debug_print(_("Writing mail folder list..."));
		gtk_ctree_pre_recursive(ctree, folderview->mailbox,
					folderview_write_cache_func, fp);
		debug_print(_("done.\n"));
		fclose(fp);
	}
	g_free(fcache);

	fcache = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, RC_DIR,
			     G_DIR_SEPARATOR_S, NEWS_FOLDER_LIST, NULL);
	if ((fp = fopen(fcache, "w")) == NULL) {
		fprintf(stderr, "%s: ", fcache);
		perror("fopen");
	} else {
		debug_print(_("Writing news folder list..."));
		gtk_ctree_pre_recursive(ctree, folderview->news,
					folderview_write_cache_func, fp);
		debug_print(_("done.\n"));
		fclose(fp);
	}
	g_free(fcache);

}

void folderview_update_msg_num(FolderView *folderview, GtkCTreeNode *row,
			       gint new, gint unread, gint total)
{
	GtkStyle *style;
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	static GtkCTreeNode *prev_row = NULL;
	FolderInfo *finfo;

	if (!row) return;

	finfo = gtk_ctree_node_get_row_data(ctree, row);
	if (!finfo) return;
	if (prev_row      == row    &&
	    finfo->new    == new    &&
	    finfo->unread == unread &&
	    finfo->total  == total)
		return;

	prev_row = row;

	/* set message number */
	finfo->new    = new;
	finfo->unread = unread;
	finfo->total  = total;
	gtk_ctree_node_set_text(ctree, row, COL_NEW, itos(new));
	gtk_ctree_node_set_text(ctree, row, COL_UNREAD, itos(unread));
	gtk_ctree_node_set_text(ctree, row, COL_TOTAL, itos(total));

	style = gtk_style_copy(gtk_widget_get_style(GTK_WIDGET(ctree)));

	if (style) {
		/* if unread messages exist, print with bold font */
		if (unread > 0)
			style->font = boldfont;

		/* if new messages exist, print with colored letter */
		if (new > 0) {
			style->fg[GTK_STATE_NORMAL]   = folderview->color_new;
			style->fg[GTK_STATE_SELECTED] = folderview->color_new;
		}

		gtk_ctree_node_set_row_style(ctree, row, style);
	}
}

void folderview_scan_folder(FolderView *folderview, const gchar *folder)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	GtkCTreeNode *node;
	DIR *dp;
	struct dirent *d;
	struct stat s;
	gchar *buf;
	gint nmessage = 0;
	gint new, unread, total;

	g_return_if_fail(folder != NULL);

	node = gtk_ctree_find_by_row_data_custom
		(ctree, NULL, (gpointer)folder,
		 (GCompareFunc)folderview_compare_path);
	if (!node) return;

	if ((dp = opendir(folder)) == NULL) {
		fprintf(stderr, "%s: ", folder);
		perror("opendir");
		return;
	}

	buf = g_strdup_printf(_("Reading folder %s ..."), folder);
	debug_print("%s\n", buf);
	STATUSBAR_PUSH(folderview->mainwin, buf);
	g_free(buf);
	GTK_EVENTS_FLUSH();

	while ((d = readdir(dp)) != NULL) {
		if (to_number(d->d_name) < 0) continue;

		buf = g_strconcat(folder, G_DIR_SEPARATOR_S, d->d_name, NULL);

		if (stat(buf, &s) < 0) {
			fprintf(stderr, "%s: ", buf);
			perror("stat");
		} else
			if (S_ISREG(s.st_mode)) nmessage++;

		g_free(buf);
	}

	closedir(dp);

	procmsg_get_mark_sum(folder, &new, &unread, &total);
	if (nmessage > total) {
		new += nmessage - total;
		unread += nmessage - total;
	}

	folderview_update_msg_num(folderview, node, new, unread, nmessage);

	STATUSBAR_POP(folderview->mainwin);
}

void folderview_scan_foreach_func(gpointer key, gpointer val, gpointer data)
{
	FolderView *folderview = (FolderView *)data;

	debug_print("folder = %s, num = %d\n",
		    (gchar *)key, GPOINTER_TO_INT(val));
	folderview_scan_folder(folderview, (gchar *)key);
	g_free(key);
}

/* rescan all folders */
void folderview_update_all(FolderView *folderview)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	MainWindow *mainwin = folderview->mainwin;

	CHDIR_RETURN_IF_FAIL(maildir);

	STATUSBAR_PUSH(mainwin, _("Rescanning all folders..."));
	main_window_cursor_wait(mainwin);

	gtk_ctree_pre_recursive(ctree, folderview->mailbox,
				GTK_CTREE_FUNC(folderview_free_data_func),
				NULL);

	gtk_clist_freeze(GTK_CLIST(ctree));
	gtk_clist_clear(GTK_CLIST(ctree));
	gtk_clist_thaw(GTK_CLIST(ctree));

	gtk_clist_freeze(GTK_CLIST(ctree));

	folderview_set_mailbox_folder(folderview);
	folderview_set_news_folder(folderview);

	/* create folder tree */
	folderview_scan_mailbox(folderview, folderview->mailbox, ".");
	gtk_ctree_sort_recursive(ctree, folderview->mailbox);
	folderview_set_special_folders(folderview);

	folderview_read_list(folderview, F_NEWSGROUP);

	folderview_write_cache(folderview);

	gtk_clist_thaw(GTK_CLIST(ctree));
	main_window_cursor_normal(mainwin);
	STATUSBAR_POP(mainwin);
}

static void folderview_set_mailbox_folder(FolderView *folderview)
{
	gchar *text[N_FOLDER_COLS] = {_("Mailbox"), "-", "-", "-"};
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	FolderInfo *finfo;

	folderview->mailbox =
		gtk_ctree_insert_node(ctree, NULL, NULL, text, FOLDER_SPACING,
				      folderxpm, folderxpmmask,
				      folderopenxpm, folderopenxpmmask,
				      FALSE, FALSE);
	finfo = g_new0(FolderInfo, 1);
	finfo->type = F_TOPMAILFOLDER;
	gtk_ctree_node_set_row_data(ctree, folderview->mailbox, finfo);
}

static void folderview_set_news_folder(FolderView *folderview)
{
	gchar *text[N_FOLDER_COLS] = {_("NetNews"), "-", "-", "-"};
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	FolderInfo *finfo;

	folderview->news =
		gtk_ctree_insert_node(ctree, NULL, NULL, text, FOLDER_SPACING,
				      folderxpm, folderxpmmask,
				      folderopenxpm, folderopenxpmmask,
				      FALSE, FALSE);
	finfo = g_new0(FolderInfo, 1);
	finfo->type = F_TOPNEWSFOLDER;
	gtk_ctree_node_set_row_data(ctree, folderview->news, finfo);
}

static GtkCTreeNode *set_special_folder(FolderView *folderview,
					GtkCTreeNode *prevnode,
					const gchar *folder,
					const gchar *newname,
					GdkPixmap *xpm, GdkBitmap *mask)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	GtkCTreeNode *node, *sibling;
	GtkCTreeNode *tnode = folderview->mailbox;

	node = gtk_ctree_find_by_row_data_custom
		(ctree, NULL, (gpointer)folder,
		 (GCompareFunc)folderview_compare_path);
	if (!node) {
		g_warning(_("%s not found.\n"), folder);
	} else {
		gtk_ctree_set_node_info(ctree, node, newname, FOLDER_SPACING,
					xpm, mask, xpm, mask, FALSE, TRUE);
		if (!prevnode)
			sibling = GTK_CTREE_ROW(tnode)->children;
		else
			sibling = GTK_CTREE_ROW(prevnode)->sibling;
		if (node != sibling)
			gtk_ctree_move(ctree, node, tnode, sibling);
	}

	return node;
}

static void folderview_set_special_folders(FolderView *folderview)
{
	GtkCTreeNode *node = NULL;

	node = set_special_folder(folderview, node, "inbox", _("Inbox"),
				  inboxxpm, inboxxpmmask);
	node = set_special_folder(folderview, node, "outbox", _("Outbox"),
				  outboxxpm, outboxxpmmask);
	node = set_special_folder(folderview, node, "draft", _("Draft"),
				  folderxpm, folderxpmmask);
	node = set_special_folder(folderview, node, "queue", _("Queue"),
				  outboxxpm, outboxxpmmask);
	node = set_special_folder(folderview, node, "trash", _("Trash box"),
				  trashxpm,  trashxpmmask);
}

static gint folderview_read_list(FolderView *folderview, FolderType type)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	gchar *text[N_FOLDER_COLS] = {NULL, NULL, NULL, NULL};
	gint retval;
	gchar dir[MAXPATHLEN];
	gchar *p;
	gchar *fcache;
	GtkCTreeNode *parent, *curnode, *topnode;
	gint new, unread, total;
	FolderInfo *finfo;
	FILE *fp;

	g_return_val_if_fail(type == F_MHFOLDER || type == F_NEWSGROUP, -1);

	fcache = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, RC_DIR,
			     G_DIR_SEPARATOR_S,
			     type == F_MHFOLDER ? MAIL_FOLDER_LIST :
			     NEWS_FOLDER_LIST, NULL);
	if ((fp = fopen(fcache, "r")) == NULL) {
		fprintf(stderr, "%s: ", fcache);
		perror("fopen");
		g_free(fcache);
		return -1;
	}
	debug_print(_("reading folder list %s ..."), fcache);
	g_free(fcache);

	if (type == F_MHFOLDER)
		topnode = folderview->mailbox;
	else
		topnode = folderview->news;

	while (fgets(dir, sizeof(dir), fp) != NULL) {
		strretchomp(dir);
		retval = fscanf(fp, "%d %d %d\n", &new, &unread, &total);
		if (retval == EOF || retval != 3) {
			g_print(_("Broken folder list cache.\n"));
			break;
		}
		if (type == F_MHFOLDER && !is_dir_exist(dir)) continue;

		/* get parent directory */
		if ((p = strrchr(dir, G_DIR_SEPARATOR)) != NULL) {
			gchar *parent_dir;

			parent_dir = g_dirname(dir);
			parent = gtk_ctree_find_by_row_data_custom
				(ctree, topnode, parent_dir,
				 (GCompareFunc)folderview_compare_path);
			if (!parent) parent = topnode;
			g_free(parent_dir);
		} else
			parent = topnode;
		gtk_ctree_expand(ctree, parent);

		text[COL_FOLDER] = g_basename(dir);
		curnode = gtk_ctree_insert_node
			(ctree, parent, NULL, text, FOLDER_SPACING,
			 folderxpm, folderxpmmask,
			 folderopenxpm, folderopenxpmmask,
			 (type == F_NEWSGROUP && parent != topnode) ?
			 TRUE : FALSE, FALSE);

		/* set folder info */
		finfo = g_new0(FolderInfo, 1);

		if (type == F_MHFOLDER)
			finfo->type = F_MHFOLDER;
		else {
			if (parent == topnode)
				finfo->type = F_NEWSSERVER;
			else
				finfo->type = F_NEWSGROUP;
		}

		finfo->path = g_strdup(dir);

		gtk_ctree_node_set_row_data(ctree, curnode, finfo);

		folderview_update_msg_num
			(folderview, curnode, new, unread, total);
	}
	fclose(fp);

	debug_print(_("done.\n"));

	return 0;
}

static void folderview_write_cache_func(GtkCTree *ctree, GtkCTreeNode *node,
					gpointer data)
{
	FILE *fp = data;
	FolderInfo *finfo;

	finfo = gtk_ctree_node_get_row_data(ctree, node);
	if (!finfo || !finfo->path) return;

	fprintf(fp, "%s\n%d %d %d\n",
		finfo->path, finfo->new, finfo->unread, finfo->total);
}

/* add folders to the tree recursively */
static void folderview_scan_mailbox(FolderView *folderview, GtkCTreeNode *node,
				    const gchar *dir)
{
	DIR *dp;
	struct dirent *d;
	struct stat s;
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	GtkCTreeNode *subnode;
	gchar *text[N_FOLDER_COLS];
	gchar *entry = NULL;
	gchar *buf;
	gint nmessage = 0;
	gint new, unread, total;

	if ((dp = opendir(dir)) == NULL) {
		perror("opendir");
		return;
	}

	buf = g_strdup_printf(_("Reading folder %s ..."), dir);
	debug_print("%s\n", buf);
	STATUSBAR_PUSH(folderview->mainwin, buf);
	g_free(buf);
	GTK_EVENTS_FLUSH();

	if (strcmp(dir, ".") != 0) {
		FolderInfo *finfo;

		finfo = g_new0(FolderInfo, 1);
		finfo->type = F_MHFOLDER;
		finfo->path = g_strdup(dir);
		gtk_ctree_node_set_row_data(ctree, node, finfo);
	}

	while ((d = readdir(dp)) != NULL) {
		if (!strcmp(d->d_name, ".") || !strcmp(d->d_name, ".."))
			continue;

		if (strcmp(dir, ".") != 0)
			entry = g_strconcat(dir, "/", d->d_name, NULL);
		else
			entry = g_strdup(d->d_name);

		if (stat(entry, &s) < 0) {
			fprintf(stderr, "%s: ", entry);
			perror("stat");
			g_free(entry);
			continue;
		}

		if (S_ISDIR(s.st_mode)) {
			text[COL_FOLDER] = d->d_name;
			text[COL_NEW] = "";
			text[COL_UNREAD] = "";
			text[COL_TOTAL] = "";
			gtk_ctree_expand(ctree, node);
			subnode = gtk_ctree_insert_node
				(ctree, node, NULL, text,
				 FOLDER_SPACING,
				 folderxpm, folderxpmmask,
				 folderopenxpm, folderopenxpmmask,
				 FALSE, FALSE);
			folderview_scan_mailbox(folderview, subnode, entry);
		} else {
			if (to_number(d->d_name) != -1) nmessage++;
		}
		g_free(entry);
	}
	closedir(dp);

	/* set the number of message */
	if (strcmp(dir, ".") != 0) {
		procmsg_get_mark_sum(dir, &new, &unread, &total);

		if (nmessage > total) {
			new += nmessage - total;
			unread += nmessage - total;
		}

		folderview_update_msg_num(folderview, node,
					  new, unread, nmessage);
	}

	STATUSBAR_POP(folderview->mainwin);
}

/* callback functions */

static void folderview_button_pressed(GtkWidget *ctree, GdkEventButton *event,
				      FolderView *folderview)
{
	GtkCList *clist = GTK_CLIST(ctree);
	gint prev_row = -1, row = -1, column = -1;
	FolderInfo *finfo;

	if (!event) return;

	if (event->button == 1) {
		folderview->open_folder = TRUE;
		return;
	}

	if (event->button == 2 || event->button == 3) {
		/* right clicked */
		if (clist->selection) {
			GtkCTreeNode *node;

			node = GTK_CTREE_NODE(clist->selection->data);
			prev_row = gtkut_ctree_get_nth_from_node
				(GTK_CTREE(ctree), node);
		}

		gtk_clist_get_selection_info(clist, event->x, event->y,
					     &row, &column);
		if (prev_row != row) {
			gtk_clist_unselect_all(clist);
			if (event->button == 2)
				folderview->open_folder = TRUE;
			gtk_clist_select_row(clist, row, column);
			if (event->button == 2)
				gtkut_clist_set_focus_row(clist, row);
		}
	}

	if (event->button != 3) return;

	finfo = gtk_clist_get_row_data(clist, row);
	if (!finfo) return;

	if (finfo->type == F_TOPMAILFOLDER ||
	    (finfo->type == F_MHFOLDER && IS_SPECIAL_FOLDER(finfo->path))) {
		menu_set_sensitive(folderview->mail_factory,
				   "/Rename folder", FALSE);
		menu_set_sensitive(folderview->mail_factory,
				   "/Delete folder", FALSE);
	} else if (finfo->type == F_MHFOLDER) {
		menu_set_sensitive(folderview->mail_factory,
				   "/Rename folder", TRUE);
		menu_set_sensitive(folderview->mail_factory,
				   "/Delete folder", TRUE);
	} else if (finfo->type == F_TOPNEWSFOLDER) {
		menu_set_sensitive(folderview->news_factory,
				   "/Subscribe to newsgroup", FALSE);
		menu_set_sensitive(folderview->news_factory,
				   "/Remove newsgroup", FALSE);
		menu_set_sensitive(folderview->news_factory,
				   "/Add news server", TRUE);
		menu_set_sensitive(folderview->news_factory,
				   "/Remove news server", FALSE);
	} else if (finfo->type == F_NEWSSERVER) {
		menu_set_sensitive(folderview->news_factory,
				   "/Subscribe to newsgroup", TRUE);
		menu_set_sensitive(folderview->news_factory,
				   "/Remove newsgroup", FALSE);
		menu_set_sensitive(folderview->news_factory,
				   "/Add news server", FALSE);
		menu_set_sensitive(folderview->news_factory,
				   "/Remove news server", TRUE);
	} else if (finfo->type == F_NEWSGROUP) {
		menu_set_sensitive(folderview->news_factory,
				   "/Subscribe to newsgroup", TRUE);
		menu_set_sensitive(folderview->news_factory,
				   "/Remove newsgroup", TRUE);
		menu_set_sensitive(folderview->news_factory,
				   "/Add news server", FALSE);
		menu_set_sensitive(folderview->news_factory,
				   "/Remove news server", FALSE);
	}

	if (finfo->type == F_TOPMAILFOLDER || finfo->type == F_MHFOLDER)
		gtk_menu_popup(GTK_MENU(folderview->mail_popup), NULL, NULL,
			       NULL, NULL, event->button, event->time);
	else
		gtk_menu_popup(GTK_MENU(folderview->news_popup), NULL, NULL,
			       NULL, NULL, event->button, event->time);
}

static void folderview_button_released(GtkWidget *ctree, GdkEventButton *event,
				       FolderView *folderview)
{
	if (!event) return;

	if (event->button == 1) {
		gtk_ctree_select(GTK_CTREE(ctree), folderview->opened);
		gtkut_ctree_set_focus_row(GTK_CTREE(ctree),
					  folderview->opened);
	}
}

static void folderview_key_pressed(GtkWidget *widget, GdkEventKey *event,
				   FolderView *folderview)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);

	if (!event) return;

	switch (event->keyval) {
	case GDK_Return:
	case GDK_space:
		folderview->open_folder = TRUE;
		gtk_ctree_select(ctree, folderview->selected);
		break;
	default:
	}
}

static void folderview_selected(GtkCTree *ctree, GtkCTreeNode *row,
				gint column, FolderView *folderview)
{
	static gboolean can_select = TRUE;
	FolderInfo *finfo;

	folderview->selected = row;

	if (folderview->opened == row) {
		folderview->open_folder = FALSE;
		return;
	}

	if (!can_select) {
		gtk_ctree_select(ctree, folderview->opened);
		gtkut_ctree_set_focus_row(ctree, folderview->opened);
		return;
	}

	if (!folderview->open_folder) return;
	folderview->open_folder = FALSE;

	finfo = gtk_ctree_node_get_row_data(ctree, row);
	if (!finfo) return;

	can_select = FALSE;

	if (finfo->path)
		debug_print(_("Folder %s is selected\n"), finfo->path);

	folderview->opened = row;

	summary_show(folderview->summaryview, finfo, FALSE);

	can_select = TRUE;
}

static void folderview_popup_close(GtkMenuShell *menu_shell,
				   FolderView *folderview)
{
	if (!folderview->opened) return;

	gtk_ctree_select(GTK_CTREE(folderview->ctree), folderview->opened);
	gtkut_ctree_set_focus_row(GTK_CTREE(folderview->ctree),
				  folderview->opened);
}

static void folderview_new_folder_cb(FolderView *folderview, guint action,
				     GtkWidget *widget)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	gchar *text[N_FOLDER_COLS] = {NULL, "0", "0", "0"};
	FolderInfo *finfo;
	gchar *new_folder;
	gchar *folder_path;
	GtkCTreeNode *node;

	if (!folderview->selected) return;

	finfo = gtk_ctree_node_get_row_data(ctree, folderview->selected);
	if (!finfo) return;

	new_folder = input_dialog(_("New folder"),
				  _("Input the name of new folder:"),
				  _("NewFolder"));
	if (!new_folder) return;

	if (strchr(new_folder, G_DIR_SEPARATOR) != NULL) {
		gchar *message;

		message = g_strdup_printf
			(_("`%c' can't be included in folder name."),
			 G_DIR_SEPARATOR);
		alertpanel(_("Invalid name"), message, NULL, NULL, NULL);

		g_free(message);
		g_free(new_folder);
		return;
	}

	if (finfo->type == F_TOPMAILFOLDER)
		folder_path = g_strdup(new_folder);
	else
		folder_path = g_strconcat(finfo->path,
					  G_DIR_SEPARATOR_S,
					  new_folder,
					  NULL);

	/* find whether the directory already exists */
	if (gtk_ctree_find_by_row_data_custom
		(ctree, folderview->selected, folder_path,
		 (GCompareFunc)folderview_compare_path) ||
	    is_dir_exist(folder_path)) {
		gchar *message;

		message = g_strdup_printf
			(_("The folder `%s' already exists."), new_folder);
		alertpanel(_("Folder exists"), message, NULL, NULL, NULL);

		g_free(message);
		g_free(new_folder);
		g_free(folder_path);
		return;
	}

	/* create directory */
	if (mkdir(folder_path, S_IRWXU) < 0) {
		fprintf(stderr, "%s: ", folder_path);
		perror("mkdir");
		g_free(new_folder);
		g_free(folder_path);
		return;
	}
	if (chmod(folder_path, S_IRWXU) < 0) {
		fprintf(stderr, "%s: ", folder_path);
		perror("chmod");
	}

	gtk_clist_freeze(GTK_CLIST(ctree));

	/* create new sub folder on CTree */
	text[COL_FOLDER] = new_folder;
	node = gtk_ctree_insert_node(ctree, folderview->selected, NULL, text,
				     FOLDER_SPACING,
				     folderxpm, folderxpmmask,
				     folderopenxpm, folderopenxpmmask,
				     FALSE, FALSE);
	gtk_ctree_expand(ctree, folderview->selected);
	finfo = g_new0(FolderInfo, 1);
	finfo->type = F_MHFOLDER;
	finfo->path = folder_path;
	gtk_ctree_node_set_row_data(ctree, node, finfo);
	gtk_ctree_sort_node(ctree, folderview->selected);
	if (GTK_CTREE_ROW(folderview->selected)->level == 1)
		folderview_set_special_folders(folderview);

	gtk_clist_thaw(GTK_CLIST(ctree));

	g_free(new_folder);

	folderview_write_cache(folderview);
}

static void folderview_rename_folder_cb(FolderView *folderview, guint action,
					GtkWidget *widget)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	FolderInfo *finfo;
	gchar *base = NULL;
	gchar *new_folder;
	gchar *folder_path;
	gchar *message;
	gchar *p;

	if (!folderview->selected) return;

	finfo = gtk_ctree_node_get_row_data(ctree, folderview->selected);
	if (!finfo || !finfo->path) return;

	if ((p = strrchr(finfo->path, G_DIR_SEPARATOR)) != NULL) {
		base = g_strndup(finfo->path, p - finfo->path);
	}

	message = g_strdup_printf(_("Input new name for `%s':"),
				  g_basename(finfo->path));
	new_folder = input_dialog(_("Rename folder"), message,
				  g_basename(finfo->path));
	g_free(message);

	if (!new_folder) {
		g_free(base);
		return;
	}

	if (strchr(new_folder, G_DIR_SEPARATOR) != NULL) {
		message = g_strdup_printf
			(_("`%c' can't be included in folder name."),
			 G_DIR_SEPARATOR);
		alertpanel(_("Invalid name"), message, NULL, NULL, NULL);

		g_free(message);
		g_free(new_folder);
		return;
	}

	if (base)
		folder_path = g_strconcat(base, G_DIR_SEPARATOR_S, new_folder,
					  NULL);
	else
		folder_path = g_strdup(new_folder);

	/* find whether the directory already exists */
	if (gtk_ctree_find_by_row_data_custom
		(ctree, GTK_CTREE_ROW(folderview->selected)->parent,
		 folder_path, (GCompareFunc)folderview_compare_path) ||
	    is_dir_exist(folder_path)) {
		message = g_strdup_printf
			(_("The folder `%s' already exists."), new_folder);
		alertpanel(_("Folder exists"), message, NULL, NULL, NULL);

		g_free(message);
		g_free(base);
		g_free(new_folder);
		g_free(folder_path);
		return;
	}

	/* rename directory */
	if (rename(finfo->path, folder_path) < 0) {
		fprintf(stderr, "%s %s: ", finfo->path, folder_path);
		perror("rename");

		g_free(base);
		g_free(new_folder);
		g_free(folder_path);
		return;
	} else {
		SummaryView *summaryview = folderview->summaryview;

		gtk_clist_freeze(GTK_CLIST(ctree));

		gtk_ctree_set_node_info(ctree, folderview->selected,
					new_folder, FOLDER_SPACING,
					folderxpm, folderxpmmask,
					folderopenxpm, folderopenxpmmask,
					FALSE, FALSE);

		g_free(finfo->path);
		finfo->path = folder_path;
		gtk_ctree_node_set_row_data(ctree, folderview->selected, finfo);

		gtk_ctree_sort_node
			(ctree, GTK_CTREE_ROW(folderview->selected)->parent);
		if (GTK_CTREE_ROW(folderview->selected)->level <= 2)
			folderview_set_special_folders(folderview);

		if (folderview->opened == folderview->selected) {
			g_free(summaryview->cur_folder);
			summaryview->cur_folder = g_strdup(folder_path);
			summary_show(summaryview, finfo, FALSE);
		}

		gtk_clist_thaw(GTK_CLIST(ctree));
	}

	g_free(base);
	g_free(new_folder);

	folderview_write_cache(folderview);
}

static void folderview_delete_folder_cb(FolderView *folderview, guint action,
					GtkWidget *widget)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	FolderInfo *finfo;
	gchar *message;
	AlertValue avalue;

	if (!folderview->selected) return;

	finfo = gtk_ctree_node_get_row_data(ctree, folderview->selected);
	if (!finfo || !finfo->path) return;

	message = g_strdup_printf
		(_("All folder(s) and message(s) under `%s'\n"
		   "will be deleted. Do you really want to delete?"),
		 g_basename(finfo->path));

	avalue = alertpanel(_("Delete folder"), message,
			    _("No"), _("Yes"), NULL);

	g_free(message);

	if (avalue != G_ALERTALTERNATE) return;

	/* deleting folder recursively */
	debug_print(_("deleting folder %s ...\n"), finfo->path);
	if (remove_dir_recursive(finfo->path) < 0) {
		g_warning(_("can't remove folder `%s'\n"), finfo->path);
		return;
	}

	if (folderview->opened == folderview->selected ||
	    gtk_ctree_is_ancestor(ctree,
				  folderview->selected,
				  folderview->opened)) {
		summary_clear_all(folderview->summaryview);
		folderview->opened = NULL;
	}

	gtk_ctree_pre_recursive(ctree, folderview->selected,
				GTK_CTREE_FUNC(folderview_free_data_func),
				NULL);
	gtk_ctree_remove_node(ctree, folderview->selected);
	folderview_write_cache(folderview);
}

static void folderview_new_group_cb(FolderView *folderview, guint action,
				    GtkWidget *widget)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	gchar *text[N_FOLDER_COLS] = {NULL, "0", "0", "0"};
	GtkCTreeNode *servernode, *node;
	FolderInfo *finfo;
	gchar *new_group, *server;

	if (!folderview->selected) return;

	finfo = gtk_ctree_node_get_row_data(ctree, folderview->selected);
	g_return_if_fail(finfo != NULL);

	new_group = input_dialog(_("Subscribe newsgroup"),
				 _("Input subscribing newsgroup:"), NULL);
	if (!new_group) return;

	if (finfo->type == F_NEWSGROUP)
		servernode = GTK_CTREE_ROW(folderview->selected)->parent;
	else
		servernode = folderview->selected;

	if (gtk_ctree_find_by_row_data_custom
		(ctree, servernode, new_group,
		 (GCompareFunc)folderview_compare_path)) {
		gchar *message;

		message = g_strdup_printf
			(_("The newsgroup `%s' already exists."), new_group);
		alertpanel(_("Group exists"), message, NULL, NULL, NULL);

		g_free(message);
		g_free(new_group);
		return;
	}

	gtk_clist_freeze(GTK_CLIST(ctree));

	text[COL_FOLDER] = new_group;
	node = gtk_ctree_insert_node(ctree, servernode, NULL, text,
				     FOLDER_SPACING,
				     folderxpm, folderxpmmask,
				     folderopenxpm, folderopenxpmmask,
				     FALSE, FALSE);
	gtk_ctree_expand(ctree, servernode);

	finfo = gtk_ctree_node_get_row_data(ctree, servernode);
	server = finfo->path;

	finfo = g_new0(FolderInfo, 1);
	finfo->type = F_NEWSGROUP;
	finfo->path = g_strconcat(server, G_DIR_SEPARATOR_S, new_group, NULL);
	g_free(new_group);
	gtk_ctree_node_set_row_data(ctree, node, finfo);
	gtk_ctree_sort_node(ctree, servernode);

	gtk_clist_thaw(GTK_CLIST(ctree));

	folderview_write_cache(folderview);
}

static void folderview_rm_group_cb(FolderView *folderview, guint action,
				   GtkWidget *widget)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	FolderInfo *finfo;
	gchar *message;
	gchar *path;
	AlertValue avalue;

	if (!folderview->selected) return;

	finfo = gtk_ctree_node_get_row_data(ctree, folderview->selected);
	if (!finfo || !finfo->path) return;

	message = g_strdup_printf(_("Really delete newsgroup `%s'?"),
				  g_basename(finfo->path));
	avalue = alertpanel(_("Delete newsgroup"), message,
			    _("No"), _("Yes"), NULL);
	g_free(message);

	if (avalue != G_ALERTALTERNATE) return;

	/* delete cache folder */
	debug_print(_("deleting newsgroup %s ...\n"), finfo->path);
	path = g_strconcat(get_news_cache_dir(), G_DIR_SEPARATOR_S,
			   finfo->path, NULL);
	if (remove_dir_recursive(path) < 0) {
		g_warning(_("can't remove folder `%s'\n"), path);
		g_free(path);
		return;
	}
	g_free(path);

	if (folderview->opened == folderview->selected) {
		summary_clear_all(folderview->summaryview);
		folderview->opened = NULL;
	}

	folderview_free_data_func(ctree, folderview->selected, NULL);
	gtk_ctree_remove_node(ctree, folderview->selected);
	folderview_write_cache(folderview);
}

static void folderview_add_server_cb(FolderView *folderview, guint action,
				     GtkWidget *widget)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	gchar *text[N_FOLDER_COLS] = {NULL, "0", "0", "0"};
	GtkCTreeNode *node;
	FolderInfo *finfo;
	gchar *new_server;

	if (!folderview->selected) return;

	finfo = gtk_ctree_node_get_row_data(ctree, folderview->selected);
	g_return_if_fail(finfo != NULL);
	g_return_if_fail(finfo->type == F_TOPNEWSFOLDER);

	new_server = input_dialog(_("Add news server"),
				  _("Input adding news server:"), NULL);
	if (!new_server) return;

	if (gtk_ctree_find_by_row_data_custom
		(ctree, folderview->selected, new_server,
		 (GCompareFunc)folderview_compare_path)) {
		gchar *message;

		message = g_strdup_printf
			(_("The news server `%s' already exists."), new_server);
		alertpanel(_("Server exists"), message, NULL, NULL, NULL);

		g_free(message);
		g_free(new_server);
		return;
	}

	gtk_clist_freeze(GTK_CLIST(ctree));

	text[COL_FOLDER] = new_server;
	node = gtk_ctree_insert_node(ctree, folderview->selected, NULL, text,
				     FOLDER_SPACING,
				     folderxpm, folderxpmmask,
				     folderopenxpm, folderopenxpmmask,
				     FALSE, FALSE);
	gtk_ctree_expand(ctree, folderview->selected);

	finfo = g_new0(FolderInfo, 1);
	finfo->type = F_NEWSSERVER;
	finfo->path = new_server;
	gtk_ctree_node_set_row_data(ctree, node, finfo);
	gtk_ctree_sort_node(ctree, folderview->selected);

	gtk_clist_thaw(GTK_CLIST(ctree));

	folderview_write_cache(folderview);
}

static void folderview_rm_server_cb(FolderView *folderview, guint action,
				    GtkWidget *widget)
{
	GtkCTree *ctree = GTK_CTREE(folderview->ctree);
	FolderInfo *finfo;
	gchar *message;
	gchar *path;
	AlertValue avalue;

	if (!folderview->selected) return;

	finfo = gtk_ctree_node_get_row_data(ctree, folderview->selected);
	if (!finfo || !finfo->path) return;

	message = g_strdup_printf(_("Really delete news server `%s'?"),
				  finfo->path);
	avalue = alertpanel(_("Delete news server"), message,
			    _("No"), _("Yes"), NULL);
	g_free(message);

	if (avalue != G_ALERTALTERNATE) return;

	/* deleting cache folder recursively */
	debug_print(_("deleting cache folder of %s ...\n"), finfo->path);
	path = g_strconcat(get_news_cache_dir(), G_DIR_SEPARATOR_S,
			   finfo->path, NULL);
	if (remove_dir_recursive(path) < 0) {
		g_warning(_("can't remove folder `%s'\n"), path);
		g_free(path);
		return;
	}
	g_free(path);

	if (folderview->opened == folderview->selected ||
	    gtk_ctree_is_ancestor(ctree,
				  folderview->selected,
				  folderview->opened)) {
		summary_clear_all(folderview->summaryview);
		folderview->opened = NULL;
	}

	gtk_ctree_pre_recursive(ctree, folderview->selected,
				GTK_CTREE_FUNC(folderview_free_data_func),
				NULL);
	gtk_ctree_remove_node(ctree, folderview->selected);
	folderview_write_cache(folderview);
}

static void folderview_free_data_func(GtkCTree *ctree, GtkCTreeNode *node,
				      gpointer data)
{
	FolderInfo *finfo = gtk_ctree_node_get_row_data(ctree, node);

	g_free(finfo->path);
	g_free(finfo);

	gtk_ctree_node_set_row_data(ctree, node, NULL);
}

static gint folderview_compare_path(gconstpointer a, gconstpointer b)
{
	const FolderInfo *finfo = a;
	const gchar *path = b;

	return path_cmp(finfo->path, path);
}
