/* inc.c */

#include <glib.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtksignal.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include "main.h"
#include "inc.h"
#include "mainwindow.h"
#include "folderview.h"
#include "prefs_common.h"
#include "prefs_account.h"
#include "account.h"
#include "procmsg.h"
#include "socket.h"
#include "pop.h"
#include "recv.h"
#include "mbox.h"
#include "utils.h"
#include "gtkutils.h"
#include "statusbar.h"
#include "manage_window.h"
#include "progress.h"
#include "filter.h"

#define MSGBUFSIZE	8192
#define TMPMSG 		".tmpmsg"

#define PROGRESS_LABEL_SET(s) \
{ \
	progress_set_label(session->progress, s); \
	GTK_EVENTS_FLUSH(); \
}

#define POP3_ABORT(val, s) \
{ \
	PROGRESS_LABEL_SET(s); \
	if (pop3_logout(pop3_sock) != PS_SUCCESS) \
		g_warning("Error occured while quitting\n"); \
	sock_close(pop3_sock); \
	return val; \
}

#define POP3_EXIT_IF_NOTOK(f, s) \
{ \
	if ((f) != PS_SUCCESS) { \
		g_warning("Error: %s\n", s); \
		POP3_ABORT(folder_table, _("Quitting")); \
	} \
}

static void inc_account_mail		(PrefsAccount	*ac_prefs,
					 gpointer	 data);

static GHashTable *inc_pop		(PrefsAccount	*ac_prefs);
static gint inc_spool			(void);

static void inc_progress_create		(POP3Session	*session);
static void inc_cancel			(GtkButton	*button,
					 gboolean	*flag);

static GHashTable *get_pop3		(POP3Session	*session,
					 PrefsAccount	*ac_prefs);
static gint get_spool			(const gchar	*dest,
					 const gchar	*mbox);

void inc_mail(MainWindow *mainwin)
{
	if (prefs_common.use_extinc && prefs_common.extinc_path) {
		gint pid;

		/* external incorporating program */
		if ((pid = fork()) < 0) {
			perror("fork");
			return;
		}

		if (pid == 0) {
			execlp(prefs_common.extinc_path,
			       g_basename(prefs_common.extinc_path),
			       NULL);

			/* this will be called when failed */
			perror("exec");
			exit(1);
		}

		/* wait until child process is terminated */
		waitpid(pid, NULL, 0);
	} else
		inc_account_mail(cur_account, mainwin);

	if (prefs_common.inc_local) inc_spool();
}

void inc_all_account_mail(MainWindow *mainwin)
{
	account_foreach(inc_account_mail, mainwin);

	if (prefs_common.inc_local) inc_spool();
}

static void inc_account_mail(PrefsAccount *ac_prefs, gpointer data)
{
	MainWindow *mainwin = (MainWindow *)data;
	GHashTable *folder_table;

	if (!ac_prefs) return;
	if (ac_prefs->protocol == A_POP3 ||
	    ac_prefs->protocol == A_APOP) {
		debug_print(_("getting new messages of account %s...\n"),
			    ac_prefs->account_name);

		folder_table = inc_pop(ac_prefs);
		if (folder_table) {
			g_hash_table_foreach(folder_table,
					     folderview_scan_foreach_func,
					     mainwin->folderview);
			g_hash_table_destroy(folder_table);
		}
	}
}

static GHashTable *inc_pop(PrefsAccount *ac_prefs)
{
	GHashTable *folder_table;
	POP3Session session;

	g_return_val_if_fail(ac_prefs != NULL, NULL);

	inc_progress_create(&session);

	folder_table = get_pop3(&session, ac_prefs);
	statusbar_pop_all();

	progress_destroy(session.progress);

	return folder_table;
}

static gint inc_spool(void)
{
	gchar *mbox, *logname;
	gint msgs;

	logname = g_get_user_name();
	mbox = g_strconcat(prefs_common.spool_path
			   ? prefs_common.spool_path : DEFAULT_SPOOL_PATH,
			   G_DIR_SEPARATOR_S, logname, NULL);
	msgs = get_spool(NULL, mbox);
	g_free(mbox);

	return msgs;
}

static void inc_progress_create(POP3Session *session)
{
	Progress *new_progress;

	g_return_if_fail(session != NULL);

	session->cancelled = FALSE;

	new_progress = progress_create();
	gtk_window_set_title(GTK_WINDOW(new_progress->window),
			     _("Retrieving new messages"));
	gtk_signal_connect(GTK_OBJECT(new_progress->cancel_btn), "clicked",
			   GTK_SIGNAL_FUNC(inc_cancel), &session->cancelled);
	manage_window_set_transient(GTK_WINDOW(new_progress->window));

	progress_set_value(new_progress, 0.0);

	gtk_widget_show_now(new_progress->window);

	session->progress = new_progress;
}

static void inc_cancel(GtkButton *button, gboolean *flag)
{
	*flag = TRUE;
	gtk_signal_disconnect_by_data(GTK_OBJECT(button), flag);
}

static GHashTable *get_pop3(POP3Session *session, PrefsAccount *ac_prefs)
{
	gint pop3_sock;
	gint count, new, bytes, num, len;
	gint msgnum = 0;
	gchar buf[MSGBUFSIZE];
	gchar prevfolder[MAXPATHLEN] = "";
	gchar *inbox;
	GHashTable *folder_table = NULL;

	g_return_val_if_fail(ac_prefs != NULL, NULL);
	g_return_val_if_fail(ac_prefs->recv_server != NULL, NULL);
	g_return_val_if_fail(*ac_prefs->recv_server != '\0', NULL);
	g_return_val_if_fail(ac_prefs->userid != NULL, NULL);
	g_return_val_if_fail(*ac_prefs->userid != '\0', NULL);
	//g_return_val_if_fail(ac_prefs->passwd != NULL, NULL);
	CHDIR_RETURN_VAL_IF_FAIL(maildir, NULL);

	inbox = (ac_prefs->inbox && *ac_prefs->inbox) ? ac_prefs->inbox
		: INBOX_DIR;

	debug_print(_("downloading messages from POP3 server %s ...\n"),
		    ac_prefs->recv_server);

	g_snprintf(buf, sizeof(buf),
		   _("Connecting to POP3 server %s"), ac_prefs->recv_server);
	PROGRESS_LABEL_SET(buf);
	if (session->cancelled) return NULL;

	if ((pop3_sock = pop3_open(ac_prefs->recv_server,
				   ac_prefs->set_popport ? ac_prefs->popport :
				   110, buf)) < 0) {
		g_warning(_("Error occured while connecting to POP3 server\n"));
		return NULL;
	}

	PROGRESS_LABEL_SET(_("Authorizing"));
	if (session->cancelled) POP3_ABORT(NULL, _("Quitting"));

	POP3_EXIT_IF_NOTOK
		(pop3_get_auth(pop3_sock,
			       ac_prefs->protocol == A_APOP ? APOP : POP3,
			       ac_prefs->userid,
			       ac_prefs->passwd ? ac_prefs->passwd : "", buf),
		 _("Authorizing"));

	PROGRESS_LABEL_SET(_("Getting number of new messages"));
	if (session->cancelled) POP3_ABORT(NULL, _("Quitting"));
	POP3_EXIT_IF_NOTOK(pop3_get_range(pop3_sock, &count, &new, &bytes),
			   _("Getting number of new messages"));

	/* create hash table to update folder tree */
	folder_table = g_hash_table_new(g_str_hash, g_str_equal);

	for (num = (ac_prefs->rmmail || ac_prefs->getall)
	     ? 1 : count - new + 1; num <= count; num++) {
		gchar *dropfolder;
		gchar *path;
		gint val;

		if (session->cancelled)
			POP3_ABORT(folder_table, _("Quitting"));
		progress_set_percentage
			(session->progress, (gfloat)num / count);
		snprintf(buf, sizeof(buf), _("Retrieving message (%d / %d)"),
			 num, count);
		PROGRESS_LABEL_SET(buf);
		POP3_EXIT_IF_NOTOK(pop3_fetch(pop3_sock, num, &len,
					      ac_prefs->rmmail), buf);

		if (recv_write_to_file(pop3_sock, TMPMSG) < 0)
			POP3_ABORT(folder_table, _("Quitting"));

		/* filter/assort the received message */
		if (ac_prefs->assort_on_recv) {
			dropfolder = filter_get_dest_folder
				(prefs_common.fltlist, TMPMSG);
			if (!dropfolder)
				dropfolder = inbox;
			debug_print(_("%d: dest folder: %s\n"),
				    num, dropfolder);
		} else
			dropfolder = inbox;

		if (strcmp2(prevfolder, dropfolder) != 0) {
			msgnum = procmsg_get_last_message_number(dropfolder);
			if (msgnum < 0) {
				g_warning(_("can't drop message into %s\n"),
					  dropfolder);
				POP3_ABORT(folder_table, _("Quitting"));
			}
		}
		msgnum++;

		path = g_strdup_printf("%s%c%d", dropfolder,
				       G_DIR_SEPARATOR, msgnum);
		if (is_file_exist(path)) {
			g_warning(_("%s exists\n"), path);
			g_free(path);
			POP3_ABORT(folder_table, _("Quitting"));
		}

		if (rename(TMPMSG, path) < 0) {
			perror("rename");
			g_warning("can't move tmpmsg to %s\n", path);
			unlink(TMPMSG);
			g_free(path);
			POP3_ABORT(folder_table, _("Quitting"));
		}
		g_free(path);

		if ((val = GPOINTER_TO_INT(g_hash_table_lookup
					  (folder_table, dropfolder))))
			g_hash_table_insert(folder_table, dropfolder,
					    GINT_TO_POINTER(val + 1));
		else
			g_hash_table_insert(folder_table, g_strdup(dropfolder),
					    GINT_TO_POINTER(1));

		if (ac_prefs->rmmail)
			POP3_EXIT_IF_NOTOK(pop3_delete(pop3_sock, num),
					   _("Deleting message"));

		strncpy(prevfolder, dropfolder, sizeof(prevfolder));
		prevfolder[sizeof(prevfolder) - 1] = '\0';
	}

	PROGRESS_LABEL_SET(_("Quitting"));
	if (pop3_logout(pop3_sock) != PS_SUCCESS)
		g_warning("Error occured while quitting\n");
	sock_close(pop3_sock);

	return folder_table;
}

static gint get_spool(const gchar *dest, const gchar *mbox)
{
	gint msgs, size;
	gint lockfd;
	gchar *tmp_mbox = "/tmp/tmpmbox";

	g_return_val_if_fail(mbox != NULL, -1);
	CHDIR_RETURN_VAL_IF_FAIL(maildir, -1);

	if (!dest)
		dest = INBOX_DIR;

	if ((size = get_file_size(mbox)) == 0) {
		debug_print(_("no messages in local mailbox.\n"));
		return 0;
	} else if (size < 0)
		return -1;

	if ((lockfd = lock_mbox(mbox, LOCK_FLOCK)) < 0)
		return -1;

	if (copy_mbox(mbox, tmp_mbox) < 0)
		return -1;

	debug_print(_("Getting new messages from %s into %s...\n"),
		    mbox, dest);
	if ((msgs = proc_mbox(dest, tmp_mbox)) < 0) {
		unlink(tmp_mbox);
		unlock_mbox(mbox, lockfd, LOCK_FLOCK);
		return -1;
	}

	unlink(tmp_mbox);
	empty_mbox(mbox);
	unlock_mbox(mbox, lockfd, LOCK_FLOCK);

	return msgs;
}
