/* main.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkrc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>
#if HAVE_LOCALE_H
#  include <locale.h>
#endif

#include "intl.h"
#include "main.h"
#include "mainwindow.h"
#include "folderview.h"
#include "summaryview.h"
#include "prefs_common.h"
#include "prefs_account.h"
#include "account.h"
#include "procmsg.h"
#include "inc.h"
#include "import.h"
#include "alertpanel.h"
#include "utils.h"

gchar *prog_version;
gchar *maildir;
gboolean debug_mode = FALSE;

#define MAKE_DIR_IF_NOT_EXIST(dir) \
{ \
	if (!is_dir_exist(dir)) { \
		if (mkdir(dir, S_IRWXU) < 0) { \
			perror("mkdir"); \
			g_error(_("can't make %s folder.\n"), dir); \
			return 1; \
		} \
		if (chmod(dir, S_IRWXU) < 0) { \
			fprintf(stdout, "%s: ", dir); \
			perror("chmod"); \
		} \
	} \
}

int main(int argc, char *argv[])
{
	gchar *userrc;
	MainWindow *mainwin;
	FolderView *folderview;

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	gtk_set_locale();
	gtk_init(&argc, &argv);

	/* parse gtkrc files */
	gtk_rc_parse("/etc/gtkrc");
	userrc = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, ".gtkrc",
			     NULL);
	gtk_rc_parse(userrc);
	g_free(userrc);
	userrc = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, ".gtk",
			     G_DIR_SEPARATOR_S, "gtkrc", NULL);
	gtk_rc_parse(userrc);
	g_free(userrc);
	userrc = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, RC_DIR,
			     G_DIR_SEPARATOR_S, "gtkrc", NULL);
	gtk_rc_parse(userrc);
	g_free(userrc);

	gtk_rc_parse("./gtkrc");

	if (argc > 1) {
		if (!strncmp(argv[1], "--debug", 7))
			debug_mode = TRUE;
	}

	prog_version = PROG_VERSION;
	maildir = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, MAIL_DIR,
			      NULL);

	CHDIR_RETURN_VAL_IF_FAIL(g_get_home_dir(), 1);

	/* backup if old rc file exists */
	if (is_file_exist(RC_DIR)) {
		if (rename(RC_DIR, RC_DIR".bak") < 0) {
			fprintf(stderr, "%s: ", RC_DIR);
			perror("rename");
		}
	}
	MAKE_DIR_IF_NOT_EXIST(RC_DIR);
	MAKE_DIR_IF_NOT_EXIST(get_news_cache_dir());

	MAKE_DIR_IF_NOT_EXIST(MAIL_DIR);
	CHDIR_RETURN_VAL_IF_FAIL(maildir, 1);
	MAKE_DIR_IF_NOT_EXIST("inbox");
	MAKE_DIR_IF_NOT_EXIST("outbox");
	MAKE_DIR_IF_NOT_EXIST("queue");
	MAKE_DIR_IF_NOT_EXIST("draft");
	MAKE_DIR_IF_NOT_EXIST("trash");

	srandom((gint)time(NULL));

	prefs_common_read_config();
	prefs_common_save_config();

	mainwin = main_window_create
		(prefs_common.sep_folder | prefs_common.sep_msg << 1);
	folderview = mainwin->folderview;

	account_read_config_all();
	account_save_config_all();

	folderview_set(folderview);

	if (prefs_common.chk_on_startup) {
		gchar *inbox;

		if (cur_account && cur_account->inbox)
			inbox = cur_account->inbox;
		else
			inbox = INBOX_DIR;

		inc_mail(mainwin);
		folderview_unselect(folderview);
		folderview->open_folder = TRUE;
		folderview_select(folderview, inbox);
	} else
		gtk_ctree_select(GTK_CTREE(folderview->ctree),
				 folderview->mailbox);

	/* ignore SIGPIPE signal */
	signal(SIGPIPE, SIG_IGN);

	gtk_main();

	return 0;
}

void app_will_exit(GtkWidget *widget, gpointer data)
{
	MainWindow *mainwin = data;

	folderview_write_cache(mainwin->folderview);
	summary_write_cache(mainwin->summaryview);
	main_window_get_size(mainwin);
	main_window_get_position(mainwin);
	prefs_common_save_config();
	account_save_config_all();
}
