/* mbox.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/file.h>
#include <ctype.h>

#include "intl.h"
#include "mbox.h"
#include "procmsg.h"
#include "utils.h"

#define MSGBUFSIZE	8192
#define TMPMSG 		".tmpmsg"

#define FPUTS_TO_TMP_ABORT_IF_FAIL(s) \
{ \
	if (fputs(s, tmp_fp) == EOF) { \
		g_warning(_("can't write to temporary file\n")); \
		fclose(tmp_fp); \
		fclose(mbox_fp); \
		return -1; \
	} \
}

gint proc_mbox(const gchar *dest, const gchar *mbox)
{
	FILE *mbox_fp;
	gchar buf[MSGBUFSIZE], from_line[MSGBUFSIZE];
	gint msgnum, msgs = 0;

	debug_print(_("Getting messages from %s into %s...\n"), mbox, dest);

	if ((mbox_fp = fopen(mbox, "r")) == NULL) {
		perror("fopen");
		return -1;
	}

	/* ignore empty lines on the head */
	do {
		if (fgets(buf, sizeof(buf), mbox_fp) == NULL) {
			g_warning(_("can't read mbox file.\n"));
			fclose(mbox_fp);
			return -1;
		}
	} while (buf[0] == '\n' || buf[0] == '\r');

	if (strncmp(buf, "From ", 5) != 0) {
		g_warning(_("invalid mbox format: %s\n"), mbox);
		fclose(mbox_fp);
		return -1;
	}

	strcpy(from_line, buf);
	if (fgets(buf, sizeof(buf), mbox_fp) == NULL) {
		g_warning(_("malformed mbox: %s\n"), mbox);
		fclose(mbox_fp);
		return -1;
	}

	if ((msgnum = procmsg_get_last_message_number(dest)) < 0) {
		g_warning(_("can't get last message number in %s\n"), dest);
		fclose(mbox_fp);
		return -1;
	}
	msgnum++;

	do {
		FILE *tmp_fp;
		gchar *path;
		gchar *startp, *endp, *rpath;
		gint empty_line;

		if ((tmp_fp = fopen(TMPMSG, "w")) == NULL) {
			perror("fopen");
			g_warning(_("can't open temporary file\n"));
			fclose(mbox_fp);
			return -1;
		}
		if (chmod(TMPMSG, S_IRUSR|S_IWUSR) < 0) {
			perror("chmod");
			g_warning(_("can't change file mode\n"));
		}

		/* convert unix From into Return-Path */
		//g_print("%s/%d:\n%s%s\n", dest, msgnum, from_line, buf);
		startp = from_line + 5;
		endp = strchr(startp, ' ');
		if (endp == NULL)
			rpath = g_strdup(startp);
		else
			rpath = g_strndup(startp, endp - startp);
		g_strstrip(rpath);
		g_snprintf(from_line, sizeof(from_line),
			   "Return-Path: %s\n", rpath);
		g_free(rpath);

		FPUTS_TO_TMP_ABORT_IF_FAIL(from_line);
		FPUTS_TO_TMP_ABORT_IF_FAIL(buf);
		from_line[0] = '\0';

		empty_line = 0;

		while (fgets(buf, sizeof(buf), mbox_fp) != NULL) {
			if (buf[0] == '\n') {
				empty_line++;
				buf[0] = '\0';
				continue;
			}

			/* From separator */
			if (empty_line > 0 && !strncmp(buf, "From ", 5)) {
				strcpy(from_line, buf);
				if (fgets(buf, sizeof(buf), mbox_fp) != NULL) {
					if (is_header_line(buf)) break;
					g_warning(_("unescaped From found:\n%s"), from_line);
				} else
					buf[0] = '\0';
			}

			if (empty_line > 0) {
				while (empty_line--)
					FPUTS_TO_TMP_ABORT_IF_FAIL("\n");
				empty_line = 0;
			}

			if (from_line[0] != '\0') {
				FPUTS_TO_TMP_ABORT_IF_FAIL(from_line);
				from_line[0] = '\0';
			}

			if (buf[0] != '\0') {
				if (!strncmp(buf, ">From ", 6)) {
					FPUTS_TO_TMP_ABORT_IF_FAIL(buf + 1);
				} else
					FPUTS_TO_TMP_ABORT_IF_FAIL(buf);

				buf[0] = '\0';
			}
		}

		if (empty_line > 0) {
			while (--empty_line)
				FPUTS_TO_TMP_ABORT_IF_FAIL("\n");
		}

		fclose(tmp_fp);

		/* move the temporary file to the destination */
		path = g_strdup_printf("%s%c%d",
				       dest, G_DIR_SEPARATOR, msgnum);
		if (rename(TMPMSG, path) < 0) {
			perror("rename");
			g_warning(_("can't move tmpmsg to %s\n"), path);
			unlink(TMPMSG);
			fclose(mbox_fp);
			g_free(path);
			return -1;
		}
		g_free(path);
		msgnum++;
		msgs++;
	} while (from_line[0] != '\0');

	debug_print(_("%d messages found.\n"), msgs);
	return msgs;
}

gint lock_mbox(const gchar *base, LockType type)
{
	gint retval = 0;

	if (type == LOCK_FILE) {
		gchar *lockfile, *locklink;
		gint retry = 0;
		FILE *lockfp;

		lockfile = g_strdup_printf("%s.%d", base, getpid());
		if ((lockfp = fopen(lockfile, "w")) == NULL) {
			perror("fopen");
			g_warning(_("can't create lock file %s\n"), lockfile);
			g_warning(_("use 'flock' instead of 'file' if possible.\n"));
			g_free(lockfile);
			return -1;
		}

		fprintf(lockfp, "%d\n", getpid());
		fclose(lockfp);

		locklink = g_strconcat(base, ".lock", NULL);
		while (link(lockfile, locklink) < 0) {
			perror("link");
			if (retry >= 10) {
				g_warning(_("can't create %s\n"), lockfile);
				unlink(lockfile);
				g_free(lockfile);
				return -1;
			}
			if (retry == 0)
				g_warning(_("mailbox is owned by another"
					    " process, waiting...\n"));
			retry++;
			sleep(5);
		}
		unlink(lockfile);
		g_free(lockfile);
	} else if (type == LOCK_FLOCK) {
		gint lockfd;

#if HAVE_FLOCK
		if ((lockfd = open(base, O_RDONLY)) < 0) {
#else
		if ((lockfd = open(base, O_RDWR)) < 0) {
#endif
			perror("open");
			g_warning(_("can't open %s\n"), base);
			return -1;
		}
#if HAVE_FLOCK
		if (flock(lockfd, LOCK_EX|LOCK_NB) < 0) {
			perror("flock");
#else
#if HAVE_LOCKF
		if (lockf(lockfd, F_TLOCK, 0) < 0) {
			perror("lockf");
#else
		{
#endif
#endif /* HAVE_FLOCK */
			g_warning(_("can't lock %s\n"), base);
			if (close(lockfd) < 0)
				perror("close");
			return -1;
		}
		retval = lockfd;
	} else {
		g_warning(_("invalid lock type\n"));
		return -1;
	}

	return retval;
}

gint unlock_mbox(const gchar *base, gint fd, LockType type)
{
	if (type == LOCK_FILE) {
		gchar *lockfile;

		lockfile = g_strconcat(base, ".lock", NULL);
		if (unlink(lockfile) < 0) {
			perror("unlink");
			g_warning(_("can't unlink lock file %s\n"), lockfile);
			g_free(lockfile);
			return -1;
		}
		g_free(lockfile);

		return 0;
	} else if (type == LOCK_FLOCK) {
#if HAVE_FLOCK
		if (flock(fd, LOCK_UN) < 0) {
			perror("flock");
#else
#if HAVE_LOCKF
		if (lockf(fd, F_ULOCK, 0) < 0) {
			perror("lockf");
#else
		{
#endif
#endif /* HAVE_FLOCK */
			g_warning(_("can't unlock %s\n"), base);
			if (close(fd) < 0)
				perror("close");
			return -1;
		}

		if (close(fd) < 0) {
			perror("close");
			return -1;
		}

		return 0;
	}

	g_warning(_("invalid lock type\n"));
	return -1;
}

gint copy_mbox(const gchar *src, const gchar *dest)
{
	return copy_file(src, dest);
}

void empty_mbox(const gchar *mbox)
{
	if (truncate(mbox, 0) < 0) {
		FILE *fp;

		perror("truncate");
		if ((fp = fopen(mbox, "w")) == NULL) {
			perror("fopen");
			g_warning(_("can't truncate mailbox to zero.\n"));
			return;
		}
		fclose(fp);
	}
}
