/* messageview.c */

#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtktext.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtksignal.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "main.h"
#include "messageview.h"
#include "summaryview.h"
#include "procheader.h"
#include "prefs_common.h"
#include "utils.h"
#include "unmime.h"

static GdkColor quote_color = {
	(gulong)0,
	(gushort)0,
	(gushort)0,
	(gushort)0xbfff
};

static GdkColor uri_color = {
	(gulong)0,
	(gushort)0,
	(gushort)0x8fff,
	(gushort)0
};

static GdkFont *msgfont;
static GdkFont *normalfont;
static GdkFont *boldfont;

static void messageview_put_header	(MessageView	*msgview,
					 FILE		*fp);
static void messageview_key_pressed	(GtkWidget	*widget,
					 GdkEventKey	*event,
					 MessageView	*msgview);

MessageView *messageview_create(void)
{
	MessageView *messageview;
	GtkWidget *scrolledwin;
	GtkWidget *text;

	debug_print(_("Creating message view...\n"));
	messageview = g_new0(MessageView, 1);

	scrolledwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwin),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	text = gtk_text_new(gtk_scrolled_window_get_hadjustment
			    (GTK_SCROLLED_WINDOW(scrolledwin)),
			    gtk_scrolled_window_get_vadjustment
			    (GTK_SCROLLED_WINDOW(scrolledwin)));
	gtk_widget_set_usize(scrolledwin, prefs_common.mainview_width, -1);
	gtk_container_add(GTK_CONTAINER(scrolledwin), text);
	gtk_signal_connect(GTK_OBJECT(text), "key_press_event",
			   GTK_SIGNAL_FUNC(messageview_key_pressed),
			   messageview);

	messageview->scrolledwin = scrolledwin;
	messageview->text = text;

	gtk_widget_show_all(scrolledwin);

	return messageview;
}

void messageview_init(MessageView *messageview)
{
	GdkColormap *colormap;
	GdkColor color[2];
	gboolean success[2];

	color[0] = quote_color;
	color[1] = uri_color;
	colormap = gdk_window_get_colormap(messageview->text->window);
	gdk_colormap_alloc_colors(colormap, color, 2, FALSE, TRUE, success);
	if (success[0] == FALSE || success[1] == FALSE)
		g_warning(_("MessageView: color allocation failed\n"));

	messageview_set_font();
}

void messageview_show(MessageView *messageview, gchar *file)
{
	gchar buf[BUFFSIZE], buf2[BUFFSIZE];
	GtkText *text = GTK_TEXT(messageview->text);
	GdkColor *fg_color;
	FILE *fp;
	gchar *p, *ep;

	if ((fp = fopen(file, "r")) == NULL) {
		perror("fopen");
		return;
	}

	gtk_text_freeze(text);
	gtk_text_backward_delete(text, gtk_text_get_length(text));

	/* if don't put these, gtk+ dies by segmentation fault (gtk+ bug?) */
	gtk_text_thaw(text);
	gtk_text_freeze(text);

	messageview_put_header(messageview, fp);

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		//conv_anytoeuc(buf2, sizeof(buf2), buf);
		conv_jistoeuc(buf2, sizeof(buf2), buf);
		conv_unreadable(buf2);
		fg_color = NULL;

		/* change color of quotation */
		if (prefs_common.enable_color &&
		    (p = strchr(buf2, '>')) != NULL) {
			/* >, foo> ... ok, <foo>, foo bar> ... not ok */
			if (p - buf2 < 40 &&
			    memchr(buf2, ' ', p - buf2) == NULL &&
			    memchr(buf2, '<', p - buf2) == NULL)
				fg_color = &quote_color;
		}

		/* parse URI */
		if (prefs_common.enable_color &&
		    ((p = strcasestr(buf2, "http://")) != NULL ||
		     (p = strcasestr(buf2, "ftp://"))  != NULL)) {
			gtk_text_insert(text, msgfont, fg_color, NULL, buf2,
					p - buf2);
			/* find end point of URI */
			for (ep = p; *ep; ep++)
				if (!isgraph(*ep) || !isascii(*ep) ||
				    strchr("()<>\"", *ep))
					break;
			gtk_text_insert(text, msgfont, &uri_color, NULL, p,
					ep - p);
			gtk_text_insert(text, msgfont, fg_color, NULL, ep, -1);
		} else
			gtk_text_insert(text, msgfont,
					fg_color, NULL, buf2, -1);
	}

	fclose(fp);
	gtk_text_thaw(text);
}

void messageview_quote_color_set(void)
{
}

void messageview_set_font(void)
{
	if (prefs_common.messagefont)
		msgfont = gdk_fontset_load(prefs_common.messagefont);
	//if (!normalfont)
	//	normalfont = gdk_fontset_load(NORMAL_FONT);
	if (!boldfont)
		boldfont = gdk_fontset_load(BOLD_FONT);
}

static void messageview_put_header(MessageView *msgview, FILE *fp)
{
	static HeaderEntry hentry[] = {{"Date: ",         NULL, FALSE},
				       {"From: ",         NULL, TRUE},
				       {"To: ",	          NULL, FALSE},
				       {"Newsgroups: ",   NULL, FALSE},
				       {"Subject: ",      NULL, TRUE},
				       {"Cc: ",	          NULL, FALSE},
				       {"Reply-To: ",     NULL, FALSE},
				       {"Followup-To: ",  NULL, FALSE},
				       {"X-Mailer: ",     NULL, TRUE},
				       {"X-Newsreader: ", NULL, TRUE},
				       {"User-Agent: ",   NULL, TRUE},
				       {"Organization: ", NULL, TRUE},
				       {NULL,	          NULL, FALSE}};
	gchar buf[BUFFSIZE], tmp[BUFFSIZE];
	GtkText *text = GTK_TEXT(msgview->text);
	gint hnum;
	HeaderEntry *hp;

	/* show specified headers */
	while ((hnum = procheader_get_one_field(buf, sizeof(buf), fp, hentry))
	       != -1) {
		hp = hentry + hnum;

		gtk_text_insert(text, boldfont, NULL, NULL, buf,
				strlen(hp->name));
		UnMimeHeader(buf);
		conv_jistoeuc(tmp, sizeof(tmp), buf + strlen(hp->name));
		conv_unreadable(tmp);
		gtk_text_insert(text, normalfont, NULL, NULL, tmp, -1);
		gtk_text_insert(text, NULL, NULL, NULL, "\n", 1);
	}

	gtk_text_insert(text, NULL, NULL, NULL, "\n", 1);
}

static void messageview_key_pressed(GtkWidget *widget, GdkEventKey *event,
				    MessageView *msgview)
{
	SummaryView *summaryview = msgview->mainwin->summaryview;

	if (!event) return;

	switch (event->keyval) {
	case GDK_Tab:
	case GDK_Home:
	case GDK_Left:
	case GDK_Up:
	case GDK_Right:
	case GDK_Down:
	case GDK_Page_Up:
	case GDK_Page_Down:
	case GDK_End:
	case GDK_Control_L:
	case GDK_Control_R:
		break;
	default:
		summary_pass_key_press_event(summaryview, event);
		gtk_widget_grab_focus(summaryview->ctree);
		break;
	}
}
