/* news.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>

#include "intl.h"
#include "news.h"
#include "nntp.h"
#include "socket.h"
#include "recv.h"
#include "procmsg.h"
#include "procheader.h"
#include "statusbar.h"
#include "utils.h"
#include "unmime.h"

#define TMPFILE		"/tmp/.tmpfile"

static GList *session_list = NULL;

static GSList *news_get_uncached_articles(NNTPSession	*session,
					  const gchar	*path,
					  gint		 cache_last,
					  gint		*rfirst,
					  gint		*rlast);
static MsgInfo *news_parse_xover	 (gchar		*xover_str);
static GSList *news_delete_old_article	 (GSList	*alist,
					  gint		first);
static void news_delete_all_article	 (const gchar	*path);
static gint news_session_find_func	 (gconstpointer	 a,
					  gconstpointer	 b);


NNTPSession *news_connection_create(const gchar *server, gushort port)
{
	gchar buf[NNTPBUFSIZE];
	NNTPSession *session;
	gint nntp_sock;

	g_return_val_if_fail(server != NULL, NULL);

	debug_print(_("creating NNTP connection...\n"));

	if ((nntp_sock = nntp_open(server, port, buf)) < 0)
		return NULL;

	session = g_new(NNTPSession, 1);
	session->server    = g_strdup(server);
	session->sock      = nntp_sock;
	session->connected = TRUE;
	session->phase     = NNTP_READY;
	session->group     = NULL;
	session->data      = NULL;

	session_list = g_list_append(session_list, session);

	return session;
}

NNTPSession *news_session_get(const gchar *server)
{
	NNTPSession *session;
	GList *list;

	list = g_list_find_custom(session_list, (gpointer)server,
				  (GCompareFunc)news_session_find_func);

	if (!list)
		session = news_connection_create(server, 119);
	else
		session = list->data;

	return session;
}

gint news_get_article_cmd(NNTPSession *session, const gchar *cmd, gint num,
			  gchar *filename)
{
	gchar *msgid;

	if (nntp_get_article(session->sock, cmd, num, &msgid) != NN_SUCCESS)
		return -1;

	debug_print("Message-Id = %s, num = %d\n", msgid, num);
	g_free(msgid);

	if (recv_write_to_file(session->sock, filename) < 0) {
		g_warning(_("can't retrieve article\n"));
		return -1;
	}

	return 0;
}

gint news_get_article(NNTPSession *session, gint num, gchar *filename)
{
	return news_get_article_cmd(session, "ARTICLE", num, filename);
}

gint news_get_header(NNTPSession *session, gint num, gchar *filename)
{
	return news_get_article_cmd(session, "HEAD", num, filename);
}

GSList *news_get_article_info(NNTPSession *session, const gchar *path,
			      gboolean use_cache)
{
	GSList *alist;

	g_return_val_if_fail(path != NULL, NULL);

	if (use_cache) {
		GSList *newlist;

		alist = procmsg_read_cache(path, M_NEWS);

		if (session) {
			gint cache_last;
			gint first, last;

			cache_last = procmsg_get_last_num_in_cache(alist);
			newlist = news_get_uncached_articles
				(session, path, cache_last, &first, &last);
			alist = news_delete_old_article(alist, first);

			alist = g_slist_concat(alist, newlist);
		}
	} else {
		alist = news_get_uncached_articles
			(session, path, 0, NULL, NULL);
		news_delete_all_article(path);
	}

	procmsg_set_flags(alist, path, M_NEWS);

	statusbar_pop_all();

	return alist;
}

static GSList *news_get_uncached_articles(NNTPSession *session,
					  const gchar *path, gint cache_last,
					  gint *rfirst, gint *rlast)
{
	gint ok;
	gint num = 0, first = 0, last = 0, begin = 0, end = 0;
	gchar buf[NNTPBUFSIZE];
	GSList *newlist = NULL;
	GSList *llast = NULL;
	MsgInfo *msginfo;

	if (rfirst) *rfirst = 0;
	if (rlast)  *rlast  = 0;

	g_return_val_if_fail(session != NULL, NULL);
	g_return_val_if_fail(path != NULL, NULL);

	ok = nntp_group(session->sock, g_basename(path), &num, &first, &last);
	if (ok != NN_SUCCESS) {
		g_warning(_("can't set group\n"));
		return NULL;
	}

	/* calculate getting overview range */
	if (first > last) {
		g_warning(_("invalid article range\n"));
		return NULL;
	}
	if (cache_last < first)
		begin = first;
	else if (last < cache_last)
		begin = first;
	else if (last == cache_last) {
		debug_print(_("no new articles.\n"));
		return NULL;
	} else
		begin = cache_last + 1;
	end = last;

#define MAX_ARTICLE 300

	if (end - begin + 1 > MAX_ARTICLE)
		begin = end - MAX_ARTICLE + 1;

	debug_print(_("getting xover %d - %d in %s...\n"), begin, end, path);
	if (nntp_xover(session->sock, begin, end) != NN_SUCCESS) {
		g_warning(_("can't get xover\n"));
		return NULL;
	}

	for (;;) {
		if (sock_read(session->sock, buf, sizeof(buf)) < 0) {
			g_warning(_("error occured while getting xover.\n"));
			return newlist;
		}

		if (buf[0] == '.' && buf[1] == '\r') break;

		msginfo = news_parse_xover(buf);
		if (!msginfo) {
			g_warning(_("invalid xover line\n"));
			continue;
		}

		msginfo->folder = g_strdup(path);
		msginfo->flags = MSG_NEW|MSG_UNREAD|MSG_NEWS;

		if (!newlist)
			llast = newlist = g_slist_append(newlist, msginfo);
		else {
			llast = g_slist_append(llast, msginfo);
			llast = llast->next;
		}
	}

	if (rfirst) *rfirst = first;
	if (rlast)  *rlast  = last;
	return newlist;
}

#define PARSE_ONE_PARAM(p, srcp) \
{ \
	p = strchr(srcp, '\t'); \
	if (!p) return NULL; \
	else \
		*p++ = '\0'; \
}

#define CONV_HEADER(dest, src) \
{ \
	UnMimeHeader(src); \
	conv_jistoeuc(dest, sizeof(dest), src); \
	conv_unreadable(dest); \
}

static MsgInfo *news_parse_xover(gchar *xover_str)
{
	MsgInfo *msginfo;
	gchar buf[NNTPBUFSIZE];
	gchar *subject, *sender, *size, *line, *date, *msgid, *ref, *tmp;
	gchar *p;
	gint num, size_int, line_int;

	PARSE_ONE_PARAM(subject, xover_str);
	PARSE_ONE_PARAM(sender, subject);
	PARSE_ONE_PARAM(date, sender);
	PARSE_ONE_PARAM(msgid, date);
	PARSE_ONE_PARAM(ref, msgid);
	PARSE_ONE_PARAM(size, ref);
	PARSE_ONE_PARAM(line, size);

	tmp = strchr(line, '\t');
	if (!tmp) tmp = strchr(line, '\r');
	if (!tmp) tmp = strchr(line, '\n');
	if (tmp) *tmp = '\0';

	num = atoi(xover_str);
	size_int = atoi(size);
	line_int = atoi(line);

	/* set MsgInfo */
	msginfo = g_new0(MsgInfo, 1);
	msginfo->msgnum = num;
	msginfo->size = size_int;

	msginfo->date = g_strdup(date);
	msginfo->date_t = procheader_date_parse(NULL, date, 0);

	CONV_HEADER(buf, sender);
	msginfo->from = g_strdup(buf);
	msginfo->fromname = procheader_get_fromname(buf);

	CONV_HEADER(buf, subject);
	msginfo->subject = g_strdup(buf);

	extract_parenthesis(msgid, '<', '>');
	remove_space(msgid);
	if (*msgid != '\0')
		msginfo->msgid = g_strdup(msgid);

	eliminate_parenthesis(ref, '(', ')');
	if ((p = strrchr(ref, '<')) != NULL) {
		extract_parenthesis(p, '<', '>');
		remove_space(p);
		if (*p != '\0')
			msginfo->inreplyto = g_strdup(p);
	}

	return msginfo;
}

static GSList *news_delete_old_article(GSList *alist, gint first)
{
	GSList *cur, *next;
	MsgInfo *msginfo;
	gchar *cachefile;

	if (first < 2) return alist;

	for (cur = alist; cur != NULL; ) {
		next = cur->next;

		msginfo = (MsgInfo *)cur->data;
		if (msginfo && msginfo->msgnum < first) {
			debug_print(_("deleting article %d...\n"),
				    msginfo->msgnum);

			cachefile = g_strdup_printf
				("%s%c%s%c%d", get_news_cache_dir(),
				 G_DIR_SEPARATOR, msginfo->folder,
				 G_DIR_SEPARATOR, msginfo->msgnum);
			if (is_file_exist(cachefile))
				unlink(cachefile);
			g_free(cachefile);

			procmsg_msginfo_free(msginfo);
			alist = g_slist_remove(alist, msginfo);
		}

		cur = next;
	}

	return alist;
}

static void news_delete_all_article(const gchar *path)
{
	DIR *dp;
	struct dirent *d;
	gchar *dir;
	gchar *file;

	dir = g_strconcat(get_news_cache_dir(), G_DIR_SEPARATOR_S,
			  path, NULL);
	if ((dp = opendir(dir)) == NULL) {
		fprintf(stderr, "%s: ", dir);
		perror("opendir");
		g_free(dir);
		return;
	}

	debug_print(_("\tDeleting all cached articles... "));

	while ((d = readdir(dp)) != NULL) {
		if (to_number(d->d_name) < 0) continue;

		file = g_strconcat(dir, G_DIR_SEPARATOR_S, d->d_name, NULL);

		if (is_file_exist(file)) {
			if (unlink(file) < 0) {
				fprintf(stderr, "%s: ", file);
				perror("unlink");
			}
		}

		g_free(file);
	}

	closedir(dp);
	g_free(dir);

	debug_print(_("done.\n"));
}

static gint news_session_find_func(gconstpointer a, gconstpointer b)
{
	const NNTPSession *session1 = a;
	const gchar *server = b;

	return strcmp2(session1->server, server);
}
