/* pop.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <unistd.h>

#include "intl.h"
#include "pop.h"
#include "socket.h"
#include "md5ify.h"
#include "prefs_account.h"
#include "utils.h"

#define POPBUFSIZE	512
#define IDLEN		128

#define PHASE_GETAUTH   0
#define PHASE_GETRANGE  1
#define PHASE_GETSIZES  2
#define PHASE_FETCH     3
#define PHASE_LOGOUT    4

static gint pop3_phase;
static gint verbose = 1;

static void pop3_gen_send(gint sock, const gchar *format, ...);
static gint pop3_gen_recv(gint sock, gchar *buf, gint size);
static gint pop3_gen_transact(gint sock, const gchar *format, ...);

gint pop3_open(const gchar *server, gushort port, gchar *buf)
{
	gint pop3_sock;

	if ((pop3_sock = sock_connect(server, port)) < 0) {
		g_warning("Can't connect to POP3 server %s:%d\n",
			  server, port);
		return -1;
	}

	if (pop3_ok(pop3_sock, buf) == PS_SUCCESS)
		return pop3_sock;
	else {
		sock_close(pop3_sock);
		return -1;
	}
}

gint pop3_get_auth(gint sock, Pop3Protocol proto, const gchar *user,
		   const gchar *pass, const gchar *greeting)
{
	gint ok;
	gchar *start, *end;
	gchar *apop_str, *md5sum;

	pop3_phase = PHASE_GETAUTH;

	switch (proto) {
	case POP3:
		ok = pop3_gen_transact(sock, "USER %s", user);
		ok = pop3_gen_transact(sock, "PASS %s", pass);
		break;
	case APOP:
		if ((start = strchr(greeting, '<')) == NULL) {
			g_warning("Required APOP timestamp not found "
				  "in greeting\n");
			return PS_AUTHFAIL;
		}

		if ((end = strchr(start, '>')) == NULL || end == start + 1) {
			g_warning("Timestamp syntax error in greeting\n");
			return PS_AUTHFAIL;
		}

		*(end + 1) = '\0';

		apop_str = g_strconcat(start, pass, NULL);
		md5sum = MD5Digest(apop_str);
		g_free(apop_str);

		ok = pop3_gen_transact(sock, "APOP %s %s", user, md5sum);
		break;
	default:
		g_warning("Undefined protocol request in POP3_auth\n");
		ok = PS_ERROR;
	}

	if (ok != 0) {
		if (ok == PS_LOCKBUSY)
			g_warning("lock busy! Is another session active?\n");
		return ok;
	}

	sleep(2);
	return PS_SUCCESS;
}

gint pop3_get_range(gint sock, gint *count, gint *new, gint *bytes)
{
	gint ok;
	gchar buf[POPBUFSIZE + 1];

	pop3_phase = PHASE_GETRANGE;

	/* get the total message count */
	pop3_gen_send(sock, "STAT");
	ok = pop3_ok(sock, buf);
	if (ok == PS_SUCCESS)
		sscanf(buf, "%d %d", count, bytes);
	else
		return ok;

	pop3_gen_send(sock, "LAST");
	ok = pop3_ok(sock, buf);

	if (ok == PS_SUCCESS) {
		gint last;

		if (sscanf(buf, "%d", &last) == 0) {
			g_warning("protocol error\n");
			return PS_ERROR;
		}
		*new = *count - last;
	} else
		/* LAST not supported */
		*new = *count;

	return PS_SUCCESS;
}

gint pop3_get_sizes(gint sock, gint count, gint *sizes)
{
	gint ok;

	if ((ok = pop3_gen_transact(sock, "LIST")) != PS_SUCCESS)
		return ok;
	else {
		gchar buf[POPBUFSIZE + 1];

		while ((ok = pop3_gen_recv(sock, buf, sizeof(buf)))
		       == PS_SUCCESS) {
			gint num, size;

			if (buf[0] == '.')
				break;
			else if (sscanf(buf, "%d %d", &num, &size) == 2)
				sizes[num - 1] = size;
		}

		return ok;
	}
}

gint pop3_fetch(gint sock, gint num, gint *len, gboolean remove_mail)
{
	gint ok;

	if (remove_mail)
		pop3_gen_send(sock, "TOP %d 99999999", num);
	else
		pop3_gen_send(sock, "RETR %d", num);
	if ((ok = pop3_ok(sock, NULL)) != 0)
		return ok;

	*len = -1;

	return PS_SUCCESS;
}

gint pop3_get_top_id(gint sock, gint num, gchar *id)
{
	gint ok;
	gboolean got_it = FALSE;
	gchar buf[POPBUFSIZE + 1];

	if ((ok = pop3_gen_transact(sock, "TOP %d 1", num)) != 0)
		return ok;

	while ((ok = pop3_gen_recv(sock, buf, sizeof(buf))) == 0) {
		if (buf[0] == '.')
			break;
		if (!got_it && !strncasecmp("Message-Id:", buf, 11)) {
			got_it = TRUE;
			/* prevent stack overflows */
			buf[IDLEN + 12] = 0;
			sscanf(buf + 12, "%s", id);
		}
	}

	return 0;
}

gint  pop3_delete(gint sock, gint num)
{
	return pop3_gen_transact(sock, "DELE %d", num);
}

gint pop3_logout(gint sock)
{
	return pop3_gen_transact(sock, "QUIT");
}

gint pop3_ok(gint sock, gchar *argbuf)
{
	gint ok;
	gchar buf[POPBUFSIZE + 1];
	gchar *bufp;

	if ((ok = pop3_gen_recv(sock, buf, sizeof(buf))) == 0) {
		bufp = buf;
		if (*bufp == '+' || *bufp == '-')
			bufp++;
		else
			return PS_PROTOCOL;

		while (isalpha(*bufp))
			bufp++;

		if (*bufp)
			*(bufp++) = '\0';

		if (!strcmp(buf, "+OK"))
			ok = PS_SUCCESS;
		else if (!strncmp(buf, "-ERR", 4)) {
			if (pop3_phase > PHASE_GETAUTH)
				ok = PS_PROTOCOL;
			else if (strstr(bufp, "lock") ||
				 strstr(bufp, "Lock") ||
				 strstr(bufp, "LOCK") ||
				 strstr(bufp, "wait"))
				ok = PS_LOCKBUSY;
			else
				ok = PS_AUTHFAIL;
			if (*bufp)
				fprintf(stderr, "%s\n", bufp);
		} else
			ok = PS_PROTOCOL;

		if (argbuf)
			strcpy(argbuf, bufp);
	}

	return ok;
}

static void pop3_gen_send(gint sock, const gchar *format, ...)
{
	gchar buf[POPBUFSIZE + 1];
	va_list args;

	va_start(args, format);
	g_vsnprintf(buf, sizeof(buf), format, args);
	va_end(args);

	if (verbose)
		log_print("POP3> %s\n", buf);

	strcat(buf, "\r\n");
	sock_write(sock, buf, strlen(buf));
}

static gint pop3_gen_recv(gint sock, gchar *buf, gint size)
{
	if (sock_read(sock, buf, size) == -1) {
		return PS_SOCKET;
	} else {
		strretchomp(buf);

		if (verbose)
			log_print("POP3< %s\n", buf);

		return PS_SUCCESS;
	}
}

static gint pop3_gen_transact(gint sock, const gchar *format, ...)
{
	gchar buf[POPBUFSIZE + 1];
	va_list args;

	va_start(args, format);
	g_vsnprintf(buf, sizeof(buf), format, args);
	va_end(args);

	if (verbose) {
		if (!strncasecmp(buf, "PASS", 4))
			log_print("POP3> PASS ********\n");
		else
			log_print("POP3> %s\n", buf);
	}

	strcat(buf, "\r\n");
	sock_write(sock, buf, strlen(buf));

	return pop3_ok(sock, NULL);
}
