/* prefs_account.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>

#include "intl.h"
#include "main.h"
#include "prefs.h"
#include "prefs_account.h"
#include "account.h"
#include "mainwindow.h"
#include "manage_window.h"
#include "gtkutils.h"
#include "utils.h"
#include "alertpanel.h"

static gboolean cancelled;

static PrefsAccount tmp_ac_prefs;

static PrefsDialog dialog;

static struct Basic {
	GtkWidget *acname_entry;
	GtkWidget *default_chkbtn;

	GtkWidget *name_entry;
	GtkWidget *addr_entry;
	GtkWidget *org_entry;

	GtkWidget *protocol_optmenu;
	GtkWidget *inbox_entry;
	GtkWidget *recvserv_entry;
	GtkWidget *smtpserv_entry;
	GtkWidget *uid_entry;
	GtkWidget *pass_entry;
} basic;

static struct Receive {
	GtkWidget *rmmail_chkbtn;
	GtkWidget *getall_chkbtn;
	GtkWidget *autochk_chkbtn;
	GtkWidget *assort_on_recv_chkbtn;
} receive;

static struct Send {
	GtkWidget *date_chkbtn;
	GtkWidget *msgid_chkbtn;

	GtkWidget *customhdr_chkbtn;
} send;

static struct Compose {
	GtkWidget *sigpath_entry;
} compose;

static struct Advanced {
	GtkWidget *smtpport_chkbtn;
	GtkWidget *smtpport_entry;
	GtkWidget *popport_chkbtn;
	GtkWidget *popport_entry;
	GtkWidget *domain_chkbtn;
	GtkWidget *domain_entry;
} advanced;

static void prefs_account_protocol_set_data_from_optmenu(PrefParam *pparam);
static void prefs_account_protocol_set_optmenu		(PrefParam *pparam);

static PrefParam param[] = {
	/* Basic */
	{"account_name", NULL, &tmp_ac_prefs.account_name, P_STRING,
	 &basic.acname_entry, prefs_set_data_from_entry, prefs_set_entry},

	{"is_default", "FALSE", &tmp_ac_prefs.is_default, P_BOOL,
	 &basic.default_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"name", NULL, &tmp_ac_prefs.name, P_STRING,
	 &basic.name_entry, prefs_set_data_from_entry, prefs_set_entry},

	{"address", NULL, &tmp_ac_prefs.address, P_STRING,
	 &basic.addr_entry, prefs_set_data_from_entry, prefs_set_entry},

	{"organization", NULL, &tmp_ac_prefs.organization, P_STRING,
	 &basic.org_entry, prefs_set_data_from_entry, prefs_set_entry},

	{"protocol", NULL, &tmp_ac_prefs.protocol, P_OTHER,
	 &basic.protocol_optmenu,
	 prefs_account_protocol_set_data_from_optmenu,
	 prefs_account_protocol_set_optmenu},

	{"receive_server", NULL, &tmp_ac_prefs.recv_server, P_STRING,
	 &basic.recvserv_entry, prefs_set_data_from_entry, prefs_set_entry},

	{"smtp_server", "localhost", &tmp_ac_prefs.smtp_server, P_STRING,
	 &basic.smtpserv_entry, prefs_set_data_from_entry, prefs_set_entry},

	{"user_id", "ENV_USER", &tmp_ac_prefs.userid, P_STRING,
	 &basic.uid_entry, prefs_set_data_from_entry, prefs_set_entry},

	{"password", NULL, &tmp_ac_prefs.passwd, P_STRING,
	 &basic.pass_entry, prefs_set_data_from_entry, prefs_set_entry},

	{"inbox", "inbox", &tmp_ac_prefs.inbox, P_STRING,
	 &basic.inbox_entry, prefs_set_data_from_entry, prefs_set_entry},

	/* Receive */
	{"remove_mail", "TRUE", &tmp_ac_prefs.rmmail, P_BOOL,
	 &receive.rmmail_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"get_all_mail", "FALSE", &tmp_ac_prefs.getall, P_BOOL,
	 &receive.getall_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"autochk_newmail", "FALSE", &tmp_ac_prefs.autochk, P_BOOL,
	 &receive.autochk_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"assort_on_receive", "FALSE", &tmp_ac_prefs.assort_on_recv, P_BOOL,
	 &receive.assort_on_recv_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	/* Send */
	{"add_date", "TRUE", &tmp_ac_prefs.add_date, P_BOOL,
	 &send.date_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"generate_msgid", "FALSE", &tmp_ac_prefs.gen_msgid, P_BOOL,
	 &send.msgid_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"add_custom_header", "FALSE", &tmp_ac_prefs.add_customhdr, P_BOOL,
	 &send.customhdr_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	/* Compose */
	{"signature_path", "~/"DEFAULT_SIGNATURE, &tmp_ac_prefs.sig_path, P_STRING,
	 &compose.sigpath_entry,
	 prefs_set_data_from_entry, prefs_set_entry},

	/* Advanced */
	{"set_smtpport", "FALSE", &tmp_ac_prefs.set_smtpport, P_BOOL,
	 &advanced.smtpport_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"smtp_port", "25", &tmp_ac_prefs.smtpport, P_USHORT,
	 &advanced.smtpport_entry,
	 prefs_set_data_from_entry, prefs_set_entry},

	{"set_popport", "FALSE", &tmp_ac_prefs.set_popport, P_BOOL,
	 &advanced.popport_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"pop_port", "110", &tmp_ac_prefs.popport, P_USHORT,
	 &advanced.popport_entry,
	 prefs_set_data_from_entry, prefs_set_entry},

	{"set_domain", "FALSE", &tmp_ac_prefs.set_domain, P_BOOL,
	 &advanced.domain_chkbtn,
	 prefs_set_data_from_toggle, prefs_set_toggle},

	{"domain", NULL, &tmp_ac_prefs.domain, P_STRING,
	 &advanced.domain_entry,
	 prefs_set_data_from_entry, prefs_set_entry},

	{NULL, NULL, NULL, P_OTHER, NULL, NULL, NULL}
};

static void prefs_account_create		(void);
static void prefs_account_basic_create		(void);
static void prefs_account_receive_create	(void);
static void prefs_account_send_create		(void);
static void prefs_account_compose_create	(void);
static void prefs_account_advanced_create	(void);

static void prefs_account_key_pressed		(GtkWidget	*widget,
						 GdkEventKey	*event,
						 gpointer	 data);
static void prefs_account_ok			(void);
static gint prefs_account_apply			(void);
static void prefs_account_cancel		(void);

#define VSPACING	12
#define BOX_BORDER	16

void prefs_account_read_config(PrefsAccount *ac_prefs, const gchar *label)
{
	g_return_if_fail(ac_prefs != NULL);
	g_return_if_fail(label != NULL);

	memset(&tmp_ac_prefs, 0, sizeof(PrefsAccount));
	prefs_read_config(param, label, ACCOUNT_RC);
	*ac_prefs = tmp_ac_prefs;
}

void prefs_account_save_config(PrefsAccount *ac_prefs, gint id)
{
	gchar *buf;

	g_return_if_fail(ac_prefs != NULL);

	tmp_ac_prefs = *ac_prefs;
	buf = g_strdup_printf("Account: %d", id);
	prefs_save_config(param, buf, ACCOUNT_RC);
}

void prefs_account_save_config_all(GList *account_list)
{
	GList *cur;
	gchar *rcpath;
	FILE *fp;
	gint id = 0;

	if (!account_list) return;

	rcpath = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S, RC_DIR,
			     G_DIR_SEPARATOR_S, ACCOUNT_RC, NULL);

	/* backup previous rc file */
	if (is_file_exist(rcpath)) {
		gchar *rcpath_bak;

		rcpath_bak = g_strconcat(rcpath, ".bak", NULL);
		if (rename(rcpath, rcpath_bak) < 0) {
			fprintf(stderr, "%s: ", rcpath);
			perror("rename");
		}
		g_free(rcpath_bak);
	}

	if ((fp = fopen(rcpath, "w")) == NULL) {
		fprintf(stderr, "%s: ", rcpath);
		perror("fopen");
		g_free(rcpath);
		return;
	}
	if (chmod(rcpath, S_IRUSR|S_IWUSR) < 0) {
		fprintf(stderr, "%s: ", rcpath);
		perror("chmod");
	}
	g_free(rcpath);

	for (cur = account_list; cur != NULL; cur = cur->next) {
		tmp_ac_prefs = *(PrefsAccount *)cur->data;
		fprintf(fp, "[Account: %d]\n", id++);
		prefs_write_param(param, fp);
		if (cur->next)
			fputc('\n', fp);
	}

	fclose(fp);
}

void prefs_account_free(PrefsAccount *ac_prefs)
{
	if (!ac_prefs) return;

	tmp_ac_prefs = *ac_prefs;
	prefs_free(param);
}

PrefsAccount *prefs_account_open(PrefsAccount *ac_prefs)
{
	static gint ac_id = 1;
	gboolean new_account = FALSE;

	debug_print(_("Opening account preferences window...\n"));

	cancelled = FALSE;

	if (!ac_prefs) {
		ac_prefs = g_new0(PrefsAccount, 1);
		new_account = TRUE;
	}

	if (!dialog.window) {
		prefs_account_create();
	}

	manage_window_set_transient(GTK_WINDOW(dialog.window));
	gtk_notebook_set_page(GTK_NOTEBOOK(dialog.notebook), 0);
	gtk_widget_grab_focus(dialog.ok_btn);

	tmp_ac_prefs = *ac_prefs;

	if (new_account) {
		gchar *buf;

		prefs_set_dialog_to_default(param);
		buf = g_strdup_printf(_("Account%d"), ac_id);
		gtk_entry_set_text(GTK_ENTRY(basic.acname_entry), buf);
		g_free(buf);
		gtk_window_set_title(GTK_WINDOW(dialog.window),
				     _("Preferences for new account"));
		gtk_widget_hide(dialog.apply_btn);
	} else {
		prefs_set_dialog(param);
		gtk_window_set_title(GTK_WINDOW(dialog.window),
				     _("Preferences for each account"));
		gtk_widget_show(dialog.apply_btn);
	}

	gtk_widget_show(dialog.window);
	gtk_main();
	gtk_widget_hide(dialog.window);

	if (cancelled && new_account) {
		g_free(ac_prefs);
		return NULL;
	} else {
		*ac_prefs = tmp_ac_prefs;
		if (new_account) ac_id++;
		return ac_prefs;
	}
}

static void prefs_account_create(void)
{
	GtkStyle *style;
	gint page = 0;

	debug_print(_("Creating account preferences window...\n"));

	/* create dialog */
	prefs_dialog_create(&dialog);
	gtk_signal_connect(GTK_OBJECT(dialog.window), "delete_event",
			   GTK_SIGNAL_FUNC(prefs_account_cancel), NULL);
	gtk_signal_connect(GTK_OBJECT(dialog.window), "key_press_event",
			   GTK_SIGNAL_FUNC(prefs_account_key_pressed), NULL);
	gtk_signal_connect(GTK_OBJECT(dialog.window), "focus_in_event",
			   GTK_SIGNAL_FUNC(manage_window_focus_in), NULL);
	gtk_signal_connect(GTK_OBJECT(dialog.ok_btn), "clicked",
	 		   GTK_SIGNAL_FUNC(prefs_account_ok), NULL);
	gtk_signal_connect(GTK_OBJECT(dialog.apply_btn), "clicked",
			   GTK_SIGNAL_FUNC(prefs_account_apply), NULL);
	gtk_signal_connect(GTK_OBJECT(dialog.cancel_btn), "clicked",
			   GTK_SIGNAL_FUNC(prefs_account_cancel), NULL);

	prefs_account_basic_create();
	SET_NOTEBOOK_LABEL(dialog.notebook, _("Basic"), page++);
	prefs_account_receive_create();
	SET_NOTEBOOK_LABEL(dialog.notebook, _("Receive"), page++);
	prefs_account_send_create();
	SET_NOTEBOOK_LABEL(dialog.notebook, _("Send"), page++);
	prefs_account_compose_create();
	SET_NOTEBOOK_LABEL(dialog.notebook, _("Compose"), page++);
	prefs_account_advanced_create();
	SET_NOTEBOOK_LABEL(dialog.notebook, _("Advanced"), page++);

	gtk_widget_show(dialog.window);

	style = gtk_widget_get_style(dialog.window);
	if (style && style->font) {
		gint width, height;

		width = gtkut_get_font_width(style->font);
		height = gtkut_get_font_height(style->font);
		gtk_widget_set_usize(dialog.window, width * 74, height * 32);
	}
}

#define MENUITEM_ADD(menu, menuitem, label, data) \
{ \
	menuitem = gtk_menu_item_new_with_label(label); \
	gtk_widget_show(menuitem); \
	gtk_menu_append(GTK_MENU(menu), menuitem); \
	if (data) \
		gtk_object_set_user_data(GTK_OBJECT(menuitem), \
					 GINT_TO_POINTER(data)); \
}

static void prefs_account_basic_create(void)
{
	GtkWidget *vbox1;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *acname_entry;
	GtkWidget *default_chkbtn;
	GtkWidget *frame1;
	GtkWidget *table1;
	GtkWidget *name_entry;
	GtkWidget *addr_entry;
	GtkWidget *org_entry;

	GtkWidget *frame2;
	GtkWidget *vbox2;
	GtkWidget *optmenu;
	GtkWidget *optmenu_menu;
	GtkWidget *menuitem;
	GtkWidget *inbox_entry;
	GtkWidget *table2;
	GtkWidget *recvserv_entry;
	GtkWidget *smtpserv_entry;
	GtkWidget *uid_entry;
	GtkWidget *pass_entry;

	vbox1 = gtk_vbox_new (FALSE, 8);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (dialog.notebook), vbox1);
	gtk_container_set_border_width (GTK_CONTAINER (vbox1), BOX_BORDER);

	hbox = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox, FALSE, FALSE, 0);

	label = gtk_label_new (_("Name of this account"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	acname_entry = gtk_entry_new ();
	gtk_widget_show (acname_entry);
	gtk_box_pack_start (GTK_BOX (hbox), acname_entry, TRUE, TRUE, 0);

	default_chkbtn = gtk_check_button_new_with_label (_("Usually used"));
	gtk_widget_show (default_chkbtn);
	gtk_box_pack_end (GTK_BOX (hbox), default_chkbtn, FALSE, FALSE, 0);

	frame1 = gtk_frame_new (_("Personal information"));
	gtk_widget_show (frame1);
	gtk_box_pack_start (GTK_BOX (vbox1), frame1, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame1), 4);
	gtk_frame_set_label_align (GTK_FRAME (frame1), 0.01, 0.5);

	table1 = gtk_table_new (3, 2, FALSE);
	gtk_widget_show (table1);
	gtk_container_add (GTK_CONTAINER (frame1), table1);
	gtk_container_set_border_width (GTK_CONTAINER (table1), 8);
	gtk_table_set_row_spacings (GTK_TABLE (table1), 8);
	gtk_table_set_col_spacings (GTK_TABLE (table1), 8);

	label = gtk_label_new (_("Full name"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table1), label, 0, 1, 0, 1,
			  GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

	label = gtk_label_new (_("Mail address"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table1), label, 0, 1, 1, 2,
			  GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

	label = gtk_label_new (_("Organization"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table1), label, 0, 1, 2, 3,
			  GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

	name_entry = gtk_entry_new ();
	gtk_widget_show (name_entry);
	gtk_table_attach (GTK_TABLE (table1), name_entry, 1, 2, 0, 1,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

	addr_entry = gtk_entry_new ();
	gtk_widget_show (addr_entry);
	gtk_table_attach (GTK_TABLE (table1), addr_entry, 1, 2, 1, 2,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

	org_entry = gtk_entry_new ();
	gtk_widget_show (org_entry);
	gtk_table_attach (GTK_TABLE (table1), org_entry, 1, 2, 2, 3,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

	frame2 = gtk_frame_new (_("Server information"));
	gtk_widget_show (frame2);
	gtk_box_pack_start (GTK_BOX (vbox1), frame2, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame2), 4);
	gtk_frame_set_label_align (GTK_FRAME (frame2), 0.01, 0.5);

	vbox2 = gtk_vbox_new (FALSE, 8);
	gtk_widget_show (vbox2);
	gtk_container_add (GTK_CONTAINER (frame2), vbox2);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 8);

	hbox = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);

	label = gtk_label_new (_("Protocol"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	optmenu = gtk_option_menu_new ();
	gtk_widget_show (optmenu);
	gtk_box_pack_start (GTK_BOX (hbox), optmenu, FALSE, FALSE, 0);

	optmenu_menu = gtk_menu_new ();

	MENUITEM_ADD (optmenu_menu, menuitem, _("POP3 (normal)"),  A_POP3 )
	MENUITEM_ADD (optmenu_menu, menuitem, _("POP3 (APOP auth)"),  A_APOP )
	MENUITEM_ADD (optmenu_menu, menuitem, _("IMAP4"), A_IMAP4)
	gtk_widget_set_sensitive(menuitem, FALSE);

	gtk_option_menu_set_menu (GTK_OPTION_MENU (optmenu), optmenu_menu);

	label = gtk_label_new (_("Inbox"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	inbox_entry = gtk_entry_new ();
	gtk_widget_show (inbox_entry);
	gtk_box_pack_start (GTK_BOX (hbox), inbox_entry, TRUE, TRUE, 0);

	table2 = gtk_table_new (3, 4, FALSE);
	gtk_widget_show (table2);
	gtk_box_pack_start (GTK_BOX (vbox2), table2, FALSE, FALSE, 0);
	gtk_table_set_row_spacings (GTK_TABLE (table2), 8);
	gtk_table_set_col_spacings (GTK_TABLE (table2), 8);

	recvserv_entry = gtk_entry_new ();
	gtk_widget_show (recvserv_entry);
	gtk_table_attach (GTK_TABLE (table2), recvserv_entry, 1, 4, 0, 1,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

	smtpserv_entry = gtk_entry_new ();
	gtk_widget_show (smtpserv_entry);
	gtk_table_attach (GTK_TABLE (table2), smtpserv_entry, 1, 4, 1, 2,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

	uid_entry = gtk_entry_new ();
	gtk_widget_show (uid_entry);
	gtk_table_attach (GTK_TABLE (table2), uid_entry, 1, 2, 2, 3,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

	pass_entry = gtk_entry_new ();
	gtk_widget_show (pass_entry);
	gtk_table_attach (GTK_TABLE (table2), pass_entry, 3, 4, 2, 3,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_entry_set_visibility (GTK_ENTRY (pass_entry), FALSE);

	label = gtk_label_new (_("Server for receiving"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table2), label, 0, 1, 0, 1,
			  GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

	label = gtk_label_new (_("SMTP server (send)"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table2), label, 0, 1, 1, 2,
			  GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

	label = gtk_label_new (_("User ID"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table2), label, 0, 1, 2, 3,
			  GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

	label = gtk_label_new (_("Password"));
	gtk_widget_show (label);
	gtk_table_attach (GTK_TABLE (table2), label, 2, 3, 2, 3,
			  0, 0, 0, 0);

	basic.acname_entry   = acname_entry;
	basic.default_chkbtn = default_chkbtn;

	basic.name_entry = name_entry;
	basic.addr_entry = addr_entry;
	basic.org_entry  = org_entry;

	basic.protocol_optmenu = optmenu;
	basic.inbox_entry      = inbox_entry;
	basic.recvserv_entry   = recvserv_entry;
	basic.smtpserv_entry   = smtpserv_entry;
	basic.uid_entry        = uid_entry;
	basic.pass_entry       = pass_entry;
}

static void prefs_account_receive_create(void)
{
	GtkWidget *vbox;
	GtkWidget *rmmail_chkbtn;
	GtkWidget *getall_chkbtn;
	GtkWidget *autochk_chkbtn;
	GtkWidget *assort_on_recv_chkbtn;

	vbox = gtk_vbox_new (FALSE, VSPACING);
	gtk_widget_show (vbox);
	gtk_container_add (GTK_CONTAINER (dialog.notebook), vbox);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), BOX_BORDER);

	rmmail_chkbtn = gtk_check_button_new_with_label
		(_("Remove messages on server when received"));
	gtk_widget_show (rmmail_chkbtn);
	gtk_box_pack_start (GTK_BOX (vbox), rmmail_chkbtn, FALSE, FALSE, 0);

	getall_chkbtn = gtk_check_button_new_with_label
		(_("Receive all messages on server"));
	gtk_widget_show (getall_chkbtn);
	gtk_box_pack_start (GTK_BOX (vbox), getall_chkbtn, FALSE, FALSE, 0);

	autochk_chkbtn =
		gtk_check_button_new_with_label (_("Auto-check new mail"));
	gtk_widget_show (autochk_chkbtn);
	gtk_box_pack_start (GTK_BOX (vbox), autochk_chkbtn, FALSE, FALSE, 0);

	assort_on_recv_chkbtn = gtk_check_button_new_with_label
		(_("Assort messages on receiving"));
	gtk_widget_show (assort_on_recv_chkbtn);
	gtk_box_pack_start (GTK_BOX (vbox), assort_on_recv_chkbtn,
			    FALSE, FALSE, 0);

	gtk_widget_set_sensitive(autochk_chkbtn, FALSE);

	receive.rmmail_chkbtn         = rmmail_chkbtn;
	receive.getall_chkbtn         = getall_chkbtn;
	receive.autochk_chkbtn        = autochk_chkbtn;
	receive.assort_on_recv_chkbtn = assort_on_recv_chkbtn;
}

static void prefs_account_send_create(void)
{
	GtkWidget *vbox;
	GtkWidget *frame;
	GtkWidget *vbox2;
	GtkWidget *date_chkbtn;
	GtkWidget *msgid_chkbtn;
	GtkWidget *hbox;
	GtkWidget *customhdr_chkbtn;
	GtkWidget *customhdr_edit_btn;

	vbox = gtk_vbox_new (FALSE, VSPACING);
	gtk_widget_show (vbox);
	gtk_container_add (GTK_CONTAINER (dialog.notebook), vbox);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), BOX_BORDER);

	frame = gtk_frame_new (_("Header"));
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
	gtk_frame_set_label_align (GTK_FRAME (frame), 0.01, 0.5);

	vbox2 = gtk_vbox_new (FALSE, 8);
	gtk_widget_show (vbox2);
	gtk_container_add (GTK_CONTAINER (frame), vbox2);
	gtk_container_set_border_width (GTK_CONTAINER (vbox2), 8);

	date_chkbtn =
		gtk_check_button_new_with_label (_("Add Date header field"));
	gtk_widget_show (date_chkbtn);
	gtk_box_pack_start (GTK_BOX (vbox2), date_chkbtn, FALSE, FALSE, 0);

	msgid_chkbtn =
		gtk_check_button_new_with_label (_("Generate Message-ID"));
	gtk_widget_show (msgid_chkbtn);
	gtk_box_pack_start (GTK_BOX (vbox2), msgid_chkbtn, FALSE, FALSE, 0);

	hbox = gtk_hbox_new (FALSE, 12);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);

	gtk_widget_set_sensitive(hbox, FALSE);

	customhdr_chkbtn =
		gtk_check_button_new_with_label (_("Add user-defined header"));
	gtk_widget_show (customhdr_chkbtn);
	gtk_box_pack_start (GTK_BOX (hbox), customhdr_chkbtn, FALSE, FALSE, 0);

	customhdr_edit_btn = gtk_button_new_with_label (_(" Edit... "));
	gtk_widget_show (customhdr_edit_btn);
	gtk_box_pack_start (GTK_BOX (hbox), customhdr_edit_btn,
			    FALSE, FALSE, 0);

	SET_TOGGLE_SENSITIVITY (customhdr_chkbtn, customhdr_edit_btn);

	send.date_chkbtn      = date_chkbtn;
	send.msgid_chkbtn     = msgid_chkbtn;
	send.customhdr_chkbtn = customhdr_chkbtn;
}

static void prefs_account_compose_create(void)
{
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *sigpath_entry;

	vbox = gtk_vbox_new (FALSE, VSPACING);
	gtk_widget_show (vbox);
	gtk_container_add (GTK_CONTAINER (dialog.notebook), vbox);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), BOX_BORDER);

	hbox = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	label = gtk_label_new (_("Signature file"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	sigpath_entry = gtk_entry_new ();
	gtk_widget_show (sigpath_entry);
	gtk_box_pack_start (GTK_BOX (hbox), sigpath_entry, TRUE, TRUE, 0);

	compose.sigpath_entry = sigpath_entry;
}

static void prefs_account_advanced_create(void)
{
	GtkWidget *vbox1;
	GtkWidget *hbox1;
	GtkWidget *checkbtn_smtpport;
	GtkWidget *entry_smtpport;
	GtkWidget *hbox2;
	GtkWidget *checkbtn_popport;
	GtkWidget *entry_popport;
	GtkWidget *hbox3;
	GtkWidget *checkbtn_domain;
	GtkWidget *entry_domain;

	vbox1 = gtk_vbox_new (FALSE, VSPACING);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (dialog.notebook), vbox1);
	gtk_container_set_border_width (GTK_CONTAINER (vbox1), BOX_BORDER);

	hbox1 = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox1);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, FALSE, 0);

	checkbtn_smtpport =
		gtk_check_button_new_with_label (_("Specify SMTP port"));
	gtk_widget_show (checkbtn_smtpport);
	gtk_box_pack_start (GTK_BOX (hbox1), checkbtn_smtpport,
			    FALSE, FALSE, 0);

	entry_smtpport = gtk_entry_new_with_max_length (5);
	gtk_widget_show (entry_smtpport);
	gtk_box_pack_start (GTK_BOX (hbox1), entry_smtpport, FALSE, FALSE, 0);
	gtk_widget_set_usize (entry_smtpport, 64, -1);
	SET_TOGGLE_SENSITIVITY(checkbtn_smtpport, entry_smtpport);

	hbox2 = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox2, FALSE, FALSE, 0);

	checkbtn_popport =
		gtk_check_button_new_with_label (_("Specify POP3 port"));
	gtk_widget_show (checkbtn_popport);
	gtk_box_pack_start (GTK_BOX (hbox2), checkbtn_popport, FALSE, FALSE, 0);

	entry_popport = gtk_entry_new_with_max_length (5);
	gtk_widget_show (entry_popport);
	gtk_box_pack_start (GTK_BOX (hbox2), entry_popport, FALSE, FALSE, 0);
	gtk_widget_set_usize (entry_popport, 64, -1);
	SET_TOGGLE_SENSITIVITY(checkbtn_popport, entry_popport);

	hbox3 = gtk_hbox_new (FALSE, 8);
	gtk_widget_show (hbox3);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox3, FALSE, FALSE, 0);

	checkbtn_domain =
		gtk_check_button_new_with_label (_("Specify domain name"));
	gtk_widget_show (checkbtn_domain);
	gtk_box_pack_start (GTK_BOX (hbox3), checkbtn_domain, FALSE, FALSE, 0);

	entry_domain = gtk_entry_new ();
	gtk_widget_show (entry_domain);
	gtk_box_pack_start (GTK_BOX (hbox3), entry_domain, TRUE, TRUE, 0);
	SET_TOGGLE_SENSITIVITY(checkbtn_domain, entry_domain);

	advanced.smtpport_chkbtn	= checkbtn_smtpport;
	advanced.smtpport_entry		= entry_smtpport;
	advanced.popport_chkbtn		= checkbtn_popport;
	advanced.popport_entry		= entry_popport;
	advanced.domain_chkbtn		= checkbtn_domain;
	advanced.domain_entry		= entry_domain;
}

static void prefs_account_key_pressed(GtkWidget *widget, GdkEventKey *event,
				      gpointer data)
{
	if (event && event->keyval == GDK_Escape)
		prefs_account_cancel();
}

static void prefs_account_ok(void)
{
	if (prefs_account_apply() == 0)
		gtk_main_quit();
}

static gint prefs_account_apply(void)
{
	gchar *msg = NULL;

	if (*gtk_entry_get_text(GTK_ENTRY(basic.addr_entry)) == '\0')
		msg = _("Mail address is not entered.");
	else if (*gtk_entry_get_text(GTK_ENTRY(basic.recvserv_entry)) == '\0')
		msg = _("Server for receiving is not entered.");
	else if (*gtk_entry_get_text(GTK_ENTRY(basic.smtpserv_entry)) == '\0')
		msg = _("SMTP server is not entered.");
	else if (*gtk_entry_get_text(GTK_ENTRY(basic.uid_entry)) == '\0')
		msg = _("User ID is not entered.");

	if (msg) {
		alertpanel(_("Error"), msg, _("OK"), NULL, NULL);
		return -1;
	}

	prefs_set_data_from_dialog(param);
	return 0;
}

static void prefs_account_cancel(void)
{
	cancelled = TRUE;
	gtk_main_quit();
}

static void prefs_account_protocol_set_data_from_optmenu(PrefParam *pparam)
{
	GtkWidget *menu;
	GtkWidget *menuitem;

	menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(*pparam->widget));
	menuitem = gtk_menu_get_active(GTK_MENU(menu));
	*((RecvProtocol *)pparam->data) = GPOINTER_TO_INT
		(gtk_object_get_user_data(GTK_OBJECT(menuitem)));
}

static void prefs_account_protocol_set_optmenu(PrefParam *pparam)
{
	RecvProtocol protocol;
	GtkOptionMenu *optmenu = GTK_OPTION_MENU(*pparam->widget);
	GtkWidget *menu;

	protocol = *((RecvProtocol *)pparam->data);
	menu = gtk_option_menu_get_menu(optmenu);

	switch (protocol) {
	case A_POP3:
		gtk_option_menu_set_history(optmenu, 0);
		break;
	case A_APOP:
		gtk_option_menu_set_history(optmenu, 1);
		break;
	case A_IMAP4:
		gtk_option_menu_set_history(optmenu, 2);
		break;
	default:
	}
}
