/* procheader.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "intl.h"
#include "procheader.h"
#include "procmsg.h"
#include "unmime.h"
#include "utils.h"

#define BUFFSIZE	8192

#define CONV_HEADER(dest, src) \
{ \
	UnMimeHeader(src); \
	conv_jistoeuc(dest, sizeof(dest), src); \
	conv_unreadable(dest); \
}

gint procheader_get_one_field(gchar *buf, gint len, FILE *fp,
			      HeaderEntry hentry[])
{
	gint nexthead;
	gint hnum = 0;
	HeaderEntry *hp = NULL;

	if (hentry != NULL) {
		/* skip non-required headers */
		do {
			if (fgets(buf, len, fp) == NULL) return -1;
			if (buf[0] == '\r' || buf[0] == '\n') return -1;
			if (buf[0] == ' ' || buf[0] == '\t') continue;

			for (hp = hentry, hnum = 0; hp->name != NULL;
			     hp++, hnum++) {
				if (!strncasecmp(hp->name, buf,
						 strlen(hp->name)))
					break;
			}
		} while (hp->name == NULL);
	} else {
		if (fgets(buf, len, fp) == NULL) return -1;
		if (buf[0] == '\r' || buf[0] == '\n') return -1;
	}

	/* unfold the specified folded line */
	if (hp && hp->unfold) {
		gboolean folded = FALSE;
		gchar *bufp = buf + strlen(buf);

		while (1) {
			nexthead = fgetc(fp);

			/* folded */
			if (nexthead == ' ' || nexthead == '\t')
				folded = TRUE;
			else if (nexthead == EOF)
				break;
			else if (folded == TRUE) {
				/* concatenate next line */
				if ((len - (bufp - buf)) <= 2) break;

				/* replace return code on the tail end
				   with space */
				*(bufp - 1) = ' ';
				*bufp++ = nexthead;
				*bufp = '\0';
				if (nexthead == '\r' || nexthead == '\n') {
					folded = FALSE;
					continue;
				}
				if (fgets(bufp, len - (bufp - buf), fp)
				    == NULL) break;
				bufp += strlen(bufp);

				folded = FALSE;
			} else {
				ungetc(nexthead, fp);
				break;
			}
		}

		/* remove trailing return code */
		strretchomp(buf);

		return hnum;
	}

	while (1) {
		nexthead = fgetc(fp);
		if (nexthead == ' ' || nexthead == '\t') {
			size_t buflen = strlen(buf);

			/* concatenate next line */
			if ((len - buflen) > 2) {
				gchar *p = buf + buflen;

				*p++ = nexthead;
				*p = '\0';
				buflen++;
				if (fgets(p, len - buflen, fp) == NULL)
					break;
			} else
				break;
		} else {
			if (nexthead != EOF)
				ungetc(nexthead, fp);
			break;
		}
	}

	/* remove trailing return code */
	strretchomp(buf);

	return hnum;
}

gchar *procheader_get_unfolded_line(gchar *buf, gint len, FILE *fp)
{
	gboolean folded = FALSE;
	gint nexthead;
	gchar *bufp;

	if (fgets(buf, len, fp) == NULL) return NULL;
	if (buf[0] == '\r' || buf[0] == '\n') return NULL;
	bufp = buf + strlen(buf);

	while (1) {
		nexthead = fgetc(fp);

		/* folded */
		if (nexthead == ' ' || nexthead == '\t')
			folded = TRUE;
		else if (nexthead == EOF)
			break;
		else if (folded == TRUE) {
			/* concatenate next line */
			if ((len - (bufp - buf)) <= 2) break;

			/* replace return code on the tail end
			   with space */
			*(bufp - 1) = ' ';
			*bufp++ = nexthead;
			*bufp = '\0';
			if (nexthead == '\r' || nexthead == '\n') {
				folded = FALSE;
				continue;
			}
			if (fgets(bufp, len - (bufp - buf), fp)
			    == NULL) break;
			bufp += strlen(bufp);

			folded = FALSE;
		} else {
			ungetc(nexthead, fp);
			break;
		}
	}

	/* remove trailing return code */
	strretchomp(buf);

	return buf;
}

GSList *procheader_get_header_list(const gchar *file)
{
	FILE *fp;
	gchar buf[BUFFSIZE], tmp[BUFFSIZE];
	gchar *p;
	GSList *hlist = NULL;
	Header *header;

	if ((fp = fopen(file, "r")) == NULL) {
		fprintf(stderr, "%s: ", file);
		perror("fopen");
		return NULL;
	}

	while (procheader_get_unfolded_line(buf, sizeof(buf), fp) != NULL) {
		if (*buf == ':') continue;
		for (p = buf; *p && *p != ' '; p++) {
			if (*p == ':' && *(p + 1) == ' ') {
				header = g_new(Header, 1);
				header->name = g_strndup(buf, p - buf);
				p += 2;
				UnMimeHeader(p);
				conv_jistoeuc(tmp, sizeof(tmp), p);
				conv_unreadable(tmp);
				header->body = g_strdup(tmp);

				hlist = g_slist_append(hlist, header);
				break;
			}
		}
	}

	fclose(fp);
	return hlist;
}

void procheader_header_list_destroy(GSList *hlist)
{
	Header *header;

	while (hlist != NULL) {
		header = hlist->data;

		g_free(header->name);
		g_free(header->body);
		g_free(header);
		hlist = g_slist_remove(hlist, header);
	}
}

void procheader_get_header_fields(FILE *fp, HeaderEntry hentry[])
{
	gchar buf[BUFFSIZE];
	HeaderEntry *hp;
	gint hnum;

	if (hentry == NULL) return;

	while ((hnum = procheader_get_one_field(buf, sizeof(buf), fp, hentry))
	       != -1) {
		hp = hentry + hnum;

		if (hp->body == NULL)
			hp->body = g_strdup(buf + strlen(hp->name));
		else if (!strcasecmp(hp->name, "To: ") ||
			 !strcasecmp(hp->name, "Cc: ")) {
			gchar *p = hp->body;
			hp->body = g_strconcat(p, ", ", buf + 4, NULL);
			g_free(p);
		}
	}
}

enum
{
	H_DATE		= 0,
	H_FROM		= 1,
	H_TO		= 2,
	H_NEWSGROUPS	= 3,
	H_SUBJECT	= 4,
	H_MSG_ID	= 5,
	H_REFERENCES	= 6,
	H_IN_REPLY_TO	= 7,
	H_SEEN		= 8
};

void procheader_parse(MsgInfo *msginfo, const gchar *file)
{
	static HeaderEntry hentry[] = {{"Date: ",		NULL, FALSE},
				       {"From: ",		NULL, TRUE},
				       {"To: ",			NULL, TRUE},
				       {"Newsgroups: ",		NULL, TRUE},
				       {"Subject: ",		NULL, TRUE},
				       {"Message-Id",		NULL, FALSE},
				       {"References: ",		NULL, FALSE},
				       {"In-Reply-To: ",	NULL, FALSE},
				       {"Seen: ",		NULL, FALSE},
				       {NULL,			NULL, FALSE}};

	FILE *fp;
	gchar buf[BUFFSIZE], tmp[BUFFSIZE];
	gchar *reference = NULL;
	gchar *p;
	gint hnum;

	if ((fp = fopen(file, "r")) == NULL) {
		fprintf(stderr, "%s: ", file);
		perror("fopen");
		return;
	}

	msginfo->inreplyto = NULL;

	while ((hnum = procheader_get_one_field(buf, sizeof(buf), fp, hentry))
	       != -1) {
		switch (hnum) {
		case H_DATE:
			if (msginfo->date) break;
			msginfo->date_t =
				procheader_date_parse(NULL, buf + 6, 0);
			msginfo->date = g_strdup(buf + 6);
			break;
		case H_FROM:
			if (msginfo->from) break;
			CONV_HEADER(tmp, buf + 6);
			msginfo->from = g_strdup(tmp);
			msginfo->fromname = procheader_get_fromname(tmp);
			break;
		case H_TO:
			CONV_HEADER(tmp, buf + 4);
			if (msginfo->to) {
				p = msginfo->to;
				msginfo->to =
					g_strconcat(p, ", ", tmp, NULL);
				g_free(p);
			} else
				msginfo->to = g_strdup(tmp);
			break;
		case H_NEWSGROUPS:
			if (msginfo->newsgroups) {
				p = msginfo->newsgroups;
				msginfo->newsgroups =
					g_strconcat(p, ",", buf + 12, NULL);
				g_free(p);
			} else
				msginfo->newsgroups = g_strdup(buf + 12);
			break;
		case H_SUBJECT:
			if (msginfo->subject) break;
			CONV_HEADER(tmp, buf + 9);
			msginfo->subject = g_strdup(tmp);
			break;
		case H_MSG_ID:
			if (msginfo->msgid) break;
			p = buf + 12;

			extract_parenthesis(p, '<', '>');
			remove_space(p);
			msginfo->msgid = g_strdup(p);
			break;
		case H_REFERENCES:
			if (!reference) {
				p = buf + 12;

				eliminate_parenthesis(p, '(', ')');
				if ((p = strrchr(p, '<')) != NULL) {
					extract_parenthesis(p, '<', '>');
					remove_space(p);
					if (*p != '\0')
						reference = g_strdup(p);
				}
			}
			break;
		case H_IN_REPLY_TO:
			if (!reference) {
				p = buf + 13;

				eliminate_parenthesis(p, '(', ')');
				extract_parenthesis(p, '<', '>');
				remove_space(p);
				if (*p != '\0')
					reference = g_strdup(buf + 13);
			}
			break;
		case H_SEEN:
		/* mnews Seen header */
			MSG_UNSET_FLAGS(msginfo->flags, MSG_NEW|MSG_UNREAD);
			break;
		default:
		}
	}
	msginfo->inreplyto = reference;

	fclose(fp);
}

gchar *procheader_get_fromname(const gchar *str)
{
	gchar *tmp, *name;

	tmp = g_strdup(str);

	if (*tmp == '\"') {
		extract_quote(tmp, '\"');
		g_strstrip(tmp);
	} else if (strchr(tmp, '<')) {
		eliminate_parenthesis(tmp, '<', '>');
		g_strstrip(tmp);
		if (*tmp == '\0') {
			strcpy(tmp, str);
			extract_parenthesis(tmp, '<', '>');
			g_strstrip(tmp);
		}
	} else if (strchr(tmp, '(')) {
		extract_parenthesis(tmp, '(', ')');
		g_strstrip(tmp);
	}

	if (*tmp == '\0')
		name = g_strdup(str);
	else
		name = g_strdup(tmp);

	g_free(tmp);
	return name;
}

time_t procheader_date_parse(gchar *dest, const gchar *src, gint len)
{
	static gchar monthstr[] = "JanFebMarAprMayJunJulAugSepOctNovDec";
	gchar weekday[4];
	gint day;
	gchar month[4];
	gint year;
	gint hh, mm, ss;
	gchar zone[6];
	gint result;
	GDateMonth dmonth;
	struct tm t;
	gchar *p;
	time_t timer;

	/* parsing date field... */
	result = sscanf(src, "%3s, %d %3s %d %2d:%2d:%2d %5s",
			weekday, &day, month, &year, &hh, &mm, &ss, zone);
	if (result != 8) {
		result = sscanf(src, "%d %3s %d %2d:%2d:%2d %5s",
				&day, month, &year, &hh, &mm, &ss, zone);
		if (result != 7) {
			ss = 0;
			result = sscanf(src, "%3s, %d %3s %d %2d:%2d %5s",
					weekday, &day, month, &year, &hh, &mm, zone);
			if (result != 7) {
				result = sscanf(src, "%d %3s %d %2d:%2d %5s",
						&day, month, &year, &hh, &mm,
						zone);
				if (result != 6) {
					g_warning("Invalid date: %s\n", src);
					if (dest && len > 0) {
						strncpy(dest, src, len - 1);
						dest[len - 1] = '\0';
					}
					return 0;
				}
			}
		}
	}

	/* Y2K compliant :) */
	if (year < 100) {
		if (year < 70)
			year += 2000;
		else
			year += 1900;
	}

	if ((p = strstr(monthstr, month)) != NULL)
		dmonth = (gint)(p - monthstr) / 3 + 1;
	else {
		g_print(_("Invalid month\n"));
		dmonth = G_DATE_BAD_MONTH;
	}

	t.tm_sec = ss;
	t.tm_min = mm;
	t.tm_hour = hh;
	t.tm_mday = day;
	t.tm_mon = dmonth - 1;
	t.tm_year = year - 1900;
	t.tm_wday = 0;
	t.tm_yday = 0;
	t.tm_isdst = 0;

	timer = mktime(&t);

	if (dest)
		procheader_date_get_localtime(dest, len, timer);

	return timer;
}

void procheader_date_get_localtime(gchar *dest, gint len, const time_t timer)
{
	static gchar *wdaystr = N_("SunMonTueWedThuFriSat");
	static gchar *tr_wday = NULL;
	struct tm *lt;
	gint wdlen;
	gchar *wday;

	if (!tr_wday)
		tr_wday = g_strdup(gettext(wdaystr));

	lt = localtime(&timer);

	wdlen = strlen(tr_wday) / 7;
	wday = g_malloc(wdlen + 1);
	strncpy(wday, tr_wday + lt->tm_wday * wdlen, wdlen);
	wday[wdlen] = '\0';

	g_snprintf(dest, len, "%d/%d/%d(%s) %02d:%02d",
		   lt->tm_year + 1900, lt->tm_mon + 1, lt->tm_mday,
		   wday, lt->tm_hour, lt->tm_min);

	g_free(wday);
}
