/* procmsg.c */

#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

#undef MEASURE_TIME

#ifdef MEASURE_TIME
#  include <sys/time.h>
#endif

#include "main.h"
#include "utils.h"
#include "procmsg.h"
#include "procheader.h"

typedef struct _FlagInfo	FlagInfo;

struct _FlagInfo
{
	guint    msgnum;
	MsgFlags flags;
};

static GSList *procmsg_get_uncached_msgs	(GHashTable	*msg_table,
						 const gchar	*folder);

static void mark_sum_func			(gpointer	 key,
						 gpointer	 value,
						 gpointer	 data);

static GHashTable *procmsg_read_mark_file	(const gchar	*folder);
static gint procmsg_cmp_msgnum			(gconstpointer	 a,
						 gconstpointer	 b);
static gint procmsg_cmp_msgnum_for_sort		(gconstpointer	 a,
						 gconstpointer	 b);
static gint procmsg_cmp_flag_msgnum		(gconstpointer	 a,
						 gconstpointer	 b);

GSList *procmsg_get_msgs(const gchar *folder, gboolean use_cache)
{
	GSList *mlist;
	GHashTable *msg_table;
#ifdef MEASURE_TIME
	struct timeval tv_before, tv_after, tv_result;

	gettimeofday(&tv_before, NULL);
#endif

	if (use_cache) {
		GSList *newlist;

		mlist = procmsg_read_cache(folder, M_MAIL);
		msg_table = procmsg_msg_hash_table_create(mlist);

		newlist = procmsg_get_uncached_msgs(msg_table, folder);
		if (msg_table)
			g_hash_table_destroy(msg_table);

		mlist = g_slist_concat(mlist, newlist);
	} else
		mlist = procmsg_get_uncached_msgs(NULL, folder);

	procmsg_set_flags(mlist, folder, M_MAIL);

#ifdef MEASURE_TIME
	gettimeofday(&tv_after, NULL);

	timersub(&tv_after, &tv_before, &tv_result);
	g_print("procmsg_get_msgs: %s: elapsed time: %ld.%06ld sec\n",
		folder, tv_result.tv_sec, tv_result.tv_usec);
#endif

	return mlist;
}

GHashTable *procmsg_msg_hash_table_create(GSList *mlist)
{
	GHashTable *msg_table;

	if (mlist == NULL) return NULL;

	msg_table = g_hash_table_new(NULL, g_direct_equal);
	procmsg_msg_hash_table_append(msg_table, mlist);

	return msg_table;
}

void procmsg_msg_hash_table_append(GHashTable *msg_table, GSList *mlist)
{
	GSList *cur;
	MsgInfo *msginfo;

	if (msg_table == NULL || mlist == NULL) return;

	for (cur = mlist; cur != NULL; cur = cur->next) {
		msginfo = (MsgInfo *)cur->data;

		g_hash_table_insert(msg_table,
				    GUINT_TO_POINTER(msginfo->msgnum),
				    msginfo);
	}
}

GSList *procmsg_read_cache(const gchar *folder, MsgType type)
{
	GSList *mlist = NULL;
	GSList *last = NULL;
	FILE *fp;
	gchar buf[BUFFSIZE];
	MsgInfo *msginfo;
	struct stat s;
	gint ver;
	guint num;
	guint len;

	if (type == M_MAIL) {
		CHDIR_RETURN_VAL_IF_FAIL(maildir, NULL);
		CHDIR_RETURN_VAL_IF_FAIL(folder, NULL);

		if ((fp = fopen(CACHE_FILE, "r")) == NULL) {
			debug_print(_("\tNo cache file\n"));
			CHDIR_RETURN_VAL_IF_FAIL(maildir, NULL);
			return NULL;
		}
	} else {
		gchar *cache_file;

		cache_file = g_strconcat(get_news_cache_dir(),
					 G_DIR_SEPARATOR_S, folder,
					 G_DIR_SEPARATOR_S, CACHE_FILE, NULL);
		if ((fp = fopen(cache_file, "r")) == NULL) {
			debug_print(_("\tNo cache file\n"));
			g_free(cache_file);
			return NULL;
		}
		g_free(cache_file);
	}

	debug_print(_("\tReading summary cache..."));

	/* compare cache version */
	if (fread(&ver, sizeof(ver), 1, fp) != 1 ||
	    CACHE_VERSION != ver) {
		debug_print(_("Cache version is different. Discarding it.\n"));
		fclose(fp);
		CHDIR_RETURN_VAL_IF_FAIL(maildir, NULL);
		return NULL;
	}

	while (fread(&num, sizeof(num), 1, fp) == 1) {
		msginfo = g_new0(MsgInfo, 1);
		msginfo->msgnum = num;
		READ_CACHE_DATA_INT(msginfo->size, fp);
		READ_CACHE_DATA_INT(msginfo->mtime, fp);
		READ_CACHE_DATA_INT(msginfo->date_t, fp);

		READ_CACHE_DATA(msginfo->fromname, fp);

		READ_CACHE_DATA(msginfo->date, fp);
		READ_CACHE_DATA(msginfo->from, fp);
		READ_CACHE_DATA(msginfo->to, fp);
		READ_CACHE_DATA(msginfo->newsgroups, fp);
		READ_CACHE_DATA(msginfo->subject, fp);
		READ_CACHE_DATA(msginfo->msgid, fp);
		READ_CACHE_DATA(msginfo->inreplyto, fp);

		if (type == M_MAIL)
			msginfo->flags = MSG_NEW|MSG_UNREAD|MSG_CACHED;
		else
			msginfo->flags =
				MSG_NEW|MSG_UNREAD|MSG_NEWS|MSG_CACHED;

		/* if the message file doesn't exist or is changed,
		   don't add the data */
		if (type == M_MAIL &&
		    (stat(itos(num), &s) < 0 ||
		     msginfo->size  != s.st_size ||
		     msginfo->mtime != s.st_mtime)) {
			procmsg_msginfo_free(msginfo);
		} else {
			msginfo->folder = g_strdup(folder);

			if (!mlist)
				last = mlist = g_slist_append(NULL, msginfo);
			else {
				last = g_slist_append(last, msginfo);
				last = last->next;
			}
		}
	}

	fclose(fp);
	debug_print(_("done.\n"));

	CHDIR_RETURN_VAL_IF_FAIL(maildir, mlist);
	return mlist;
}

static GSList *procmsg_get_uncached_msgs(GHashTable *msg_table,
					 const gchar *folder)
{
	DIR *dp;
	struct dirent *d;
	struct stat s;
	GSList *newlist = NULL;
	GSList *last = NULL;
	MsgInfo *msginfo;
	gint n_newmsg = 0;
	gint num;

	CHDIR_RETURN_VAL_IF_FAIL(maildir, NULL);
	CHDIR_RETURN_VAL_IF_FAIL(folder, NULL);

	if ((dp = opendir(".")) == NULL) {
		g_print("%s: ", folder);
		perror("opendir");
		return NULL;
	}

	debug_print(_("\tSearching uncached messages... "));

	if (msg_table) {
		while ((d = readdir(dp)) != NULL) {
			if ((num = to_number(d->d_name)) < 0) continue;
			if (stat(d->d_name, &s) < 0) {
				perror("stat");
				continue;
			}
			if (!S_ISREG(s.st_mode)) continue;

			msginfo = g_hash_table_lookup
				(msg_table, GUINT_TO_POINTER(num));

			if (!msginfo) {
				/* not found in the cache (uncached message) */
				msginfo = g_new0(MsgInfo, 1);
				procmsg_msginfo_set(msginfo, d->d_name, folder,
						    MSG_NEW|MSG_UNREAD);
				if (!newlist)
					last = newlist =
						g_slist_append(NULL, msginfo);
				else {
					last = g_slist_append(last, msginfo);
					last = last->next;
				}
				n_newmsg++;
			}
		}
	} else {
		/* discard all previous cache */
		while ((d = readdir(dp)) != NULL) {
			if (to_number(d->d_name) < 0) continue;
			if (stat(d->d_name, &s) < 0) {
				perror("stat");
				continue;
			}
			if (!S_ISREG(s.st_mode)) continue;

			msginfo = g_new0(MsgInfo, 1);
			procmsg_msginfo_set(msginfo, d->d_name, folder,
					    MSG_NEW|MSG_UNREAD);
			if (!newlist)
				last = newlist =
					g_slist_append(NULL, msginfo);
			else {
				last = g_slist_append(last, msginfo);
				last = last->next;
			}
			n_newmsg++;
		}
	}

	closedir(dp);

	if (n_newmsg)
		debug_print(_("%d uncached message(s) found.\n"), n_newmsg);
	else
		debug_print(_("done.\n"));

	/* sort new messages in numerical order */
	if (newlist) {
		debug_print(_("\tSorting uncached messages in numerical order... "));
		newlist = g_slist_sort
			(newlist, (GCompareFunc)procmsg_cmp_msgnum_for_sort);
		debug_print(_("done.\n"));
	}

	CHDIR_RETURN_VAL_IF_FAIL(maildir, newlist);
	return newlist;
}

void procmsg_set_flags(GSList *mlist, const gchar *folder, MsgType type)
{
	GSList *cur, *tmp;
	gint newmsg = 0;
	gchar *markdir;
	MsgInfo *msginfo;
	GHashTable *mark_table;
	MsgFlags flags;

	if (!mlist) return;
	g_return_if_fail(folder != NULL);

	debug_print(_("\tMarking the messages..."));

	markdir = g_strconcat(type == M_MAIL ? maildir : get_news_cache_dir(),
			      G_DIR_SEPARATOR_S, folder, NULL);
	mark_table = procmsg_read_mark_file(markdir);
	g_free(markdir);

	if (!mark_table) return;

	for (cur = mlist; cur != NULL; cur = cur->next) {
		msginfo = (MsgInfo *)cur->data;

		flags = GPOINTER_TO_UINT(g_hash_table_lookup(mark_table,
					 GUINT_TO_POINTER(msginfo->msgnum)));

		if (flags != 0) {
			/* flag found */
			/* preserve cached flag */
			msginfo->flags &= MSG_CACHED;
			msginfo->flags |= (flags & ~MSG_CACHED);
			if (type == M_NEWS)
				msginfo->flags |= MSG_NEWS;
		} else {
			/* not found (new message) */
			if (newmsg == 0) {
				for (tmp = mlist; tmp != cur; tmp = tmp->next)
					MSG_UNSET_FLAGS
						(((MsgInfo *)tmp->data)->flags,
						 MSG_NEW);
			}
			newmsg++;
		}
	}

	debug_print(_("done.\n"));
	if (newmsg)
		debug_print(_("\t%d new message(s)\n"), newmsg);

	g_hash_table_destroy(mark_table);
}

gint procmsg_get_last_num_in_cache(GSList *mlist)
{
	GSList *cur;
	MsgInfo *msginfo;
	gint last = 0;

	if (mlist == NULL) return 0;

	for (cur = mlist; cur != NULL; cur = cur->next) {
		msginfo = (MsgInfo *)cur->data;
		if (msginfo && msginfo->msgnum > last)
			last = msginfo->msgnum;
	}

	return last;
}

void procmsg_write_cache(MsgInfo *msginfo, FILE *fp)
{
	WRITE_CACHE_DATA_INT(msginfo->msgnum, fp);
	WRITE_CACHE_DATA_INT(msginfo->size, fp);
	WRITE_CACHE_DATA_INT(msginfo->mtime, fp);
	WRITE_CACHE_DATA_INT(msginfo->date_t, fp);

	WRITE_CACHE_DATA(msginfo->fromname, fp);

	WRITE_CACHE_DATA(msginfo->date, fp);
	WRITE_CACHE_DATA(msginfo->from, fp);
	WRITE_CACHE_DATA(msginfo->to, fp);
	WRITE_CACHE_DATA(msginfo->newsgroups, fp);
	WRITE_CACHE_DATA(msginfo->subject, fp);
	WRITE_CACHE_DATA(msginfo->msgid, fp);
	WRITE_CACHE_DATA(msginfo->inreplyto, fp);
}

void procmsg_write_flags(MsgInfo *msginfo, FILE *fp)
{
	MsgFlags flags;

	flags = msginfo->flags & (MSG_NEW|MSG_UNREAD|MSG_IMPORTANT);

	WRITE_CACHE_DATA_INT(msginfo->msgnum, fp);
	WRITE_CACHE_DATA_INT(flags, fp);
}

struct MarkSum {
	gint *new;
	gint *unread;
	gint *total;
};

static void mark_sum_func(gpointer key, gpointer value, gpointer data)
{
	MsgFlags flags = GPOINTER_TO_UINT(value);
	struct MarkSum *marksum = data;

	if (MSG_IS_NEW(flags)) (*marksum->new)++;
	if (MSG_IS_UNREAD(flags)) (*marksum->unread)++;
	(*marksum->total)++;
}

void procmsg_get_mark_sum(const gchar *folder,
			  gint *new, gint *unread, gint *total)
{
	GHashTable *mark_table;
	struct MarkSum marksum;

	*new = *unread = *total = 0;
	marksum.new    = new;
	marksum.unread = unread;
	marksum.total  = total;

	mark_table = procmsg_read_mark_file(folder);

	if (mark_table) {
		g_hash_table_foreach(mark_table, mark_sum_func, &marksum);
		g_hash_table_destroy(mark_table);
	}
}

static GHashTable *procmsg_read_mark_file(const gchar *folder)
{
	FILE *fp;
	GHashTable *mark_table = NULL;
	gint num;
	MsgFlags flags;

	if ((fp = procmsg_open_mark_file(folder, FALSE)) == NULL)
		return NULL;

	mark_table = g_hash_table_new(NULL, g_direct_equal);

	while (fread(&num, sizeof(num), 1, fp) == 1) {
		if (fread(&flags, sizeof(flags), 1, fp) != 1) break;
		MSG_SET_FLAGS(flags, MSG_CACHED);

		g_hash_table_insert(mark_table,
				    GUINT_TO_POINTER(num),
				    GUINT_TO_POINTER(flags));
	}

	fclose(fp);
	return mark_table;
}

FILE *procmsg_open_mark_file(const gchar *folder, gboolean append)
{
	gchar *markfile;
	FILE *fp;
	gint ver;

	markfile = g_strconcat(folder, G_DIR_SEPARATOR_S, MARK_FILE, NULL);

	if ((fp = fopen(markfile, "r")) == NULL)
		debug_print(_("Mark file not found.\n"));
	else if (fread(&ver, sizeof(ver), 1, fp) != 1 || MARK_VERSION != ver) {
		debug_print(_("Mark version is different (%d != %d). "
			      "Discarding it.\n"), ver, MARK_VERSION);
		fclose(fp);
		fp = NULL;
	}

	/* read mode */
	if (append == FALSE) {
		g_free(markfile);
		return fp;
	}

	if (fp) {
		/* reopen with append mode */
		fclose(fp);
		if ((fp = fopen(markfile, "a")) == NULL)
			g_warning(_("Can't open mark file with append mode.\n"));
	} else {
		/* open with overwrite mode if mark file doesn't exist or
		   version is different */
		if ((fp = fopen(markfile, "w")) == NULL)
			g_warning(_("Can't open mark file with write mode.\n"));
		else {
			ver = MARK_VERSION;
			WRITE_CACHE_DATA_INT(ver, fp);
		}
	}

	g_free(markfile);
	return fp;
}

gint procmsg_get_last_message_number(const gchar *dir)
{
	gint max = 0, num;
	DIR *dp;
	struct dirent *d;

	CHDIR_RETURN_VAL_IF_FAIL(maildir, -1);
	CHDIR_RETURN_VAL_IF_FAIL(dir, -1);

	if ((dp = opendir(".")) == NULL) {
		g_print("%s: ", dir);
		perror("opendir");
		CHDIR_RETURN_VAL_IF_FAIL(maildir, -1);
		return -1;
	}

	while ((d = readdir(dp)) != NULL) {
		if ((num = to_number(d->d_name)) < 0) continue;
		if (max < num)
			max = num;
	}

	closedir(dp);

	CHDIR_RETURN_VAL_IF_FAIL(maildir, -1);
	debug_print(_("Last number in dir %s = %d\n"), dir, max);
	return max;
}

/* copy a message into the destdir and return its number */
gint procmsg_copy_message(const gchar *destdir, const gchar *src)
{
	gint num;
	gchar *destfile;

	num = procmsg_get_last_message_number(destdir);
	if (num < 0)
		return -1;

	destfile = g_strdup_printf("%s%c%d",
				   destdir, G_DIR_SEPARATOR, num + 1);
	if (copy_file(src, destfile) < 0) {
		g_warning(_("can't copy message %s to %s\n"), src, destfile);
		g_free(destfile);
		return -1;
	}

	g_free(destfile);
	return num + 1;
}

void procmsg_move_messages(const gchar *from, GSList *mlist)
{
	GSList *cur, *movelist = NULL;
	MsgInfo *msginfo;
	gchar *to_folder = NULL;

	g_return_if_fail(from != NULL);
	if (!mlist) return;

	for (cur = mlist; cur != NULL; cur = cur->next) {
		msginfo = (MsgInfo *)cur->data;
		if (!to_folder) {
			to_folder = g_strdup(msginfo->to_folder);
			movelist = g_slist_append(movelist, msginfo);
		} else if (!strcmp(to_folder, msginfo->to_folder)) {
			movelist = g_slist_append(movelist, msginfo);
		} else {
			procmsg_move_messages_with_dest
				(to_folder, from, movelist);
			slist_remove_all(movelist);
			movelist = NULL;
			g_free(to_folder);
			to_folder = g_strdup(msginfo->to_folder);
			movelist = g_slist_append(movelist, msginfo);
		}
	}

	if (movelist) {
		procmsg_move_messages_with_dest(to_folder, from, movelist);
		slist_remove_all(movelist);
		g_free(to_folder);
	}
}

void procmsg_move_messages_with_dest(const gchar *to, const gchar *from,
				     GSList *mlist)
{
	GSList *cur;
	MsgInfo *msginfo, newmsginfo;
	guint new_num;
	gchar *to_dir;
	gchar *to_path;
	FILE *fp;

	g_return_if_fail(to    != NULL);
	g_return_if_fail(from  != NULL);
	g_return_if_fail(mlist != NULL);

	to_dir = g_strconcat(maildir, G_DIR_SEPARATOR_S, to, NULL);
	new_num = procmsg_get_last_message_number(to_dir) + 1;
	if (new_num < 1)
		return;

	CHDIR_RETURN_IF_FAIL(maildir);
	CHDIR_RETURN_IF_FAIL(from);

	if ((fp = procmsg_open_mark_file(to_dir, TRUE)) == NULL)
		g_warning(_("Can't open mark file.\n"));

	for (cur = mlist; cur != NULL; cur = cur->next) {
		msginfo = (MsgInfo *)cur->data;
		to_path = g_strdup_printf("%s%c%d", to_dir, G_DIR_SEPARATOR,
					  new_num);
		debug_print(_("Moving message %s/%d to %s/%d ...\n"),
			    from, msginfo->msgnum, to, new_num);
		if (rename(itos(msginfo->msgnum), to_path) < 0)
			perror("rename");
		else if (fp) {
			newmsginfo.msgnum = new_num;
			if (!strcmp(to, OUTBOX_DIR) ||
			    !strcmp(to, QUEUE_DIR)  ||
			    !strcmp(to, DRAFT_DIR)  ||
			    !strcmp(to, TRASH_DIR))
				newmsginfo.flags = 0;
			else
				newmsginfo.flags = msginfo->flags;
			procmsg_write_flags(&newmsginfo, fp);
		}
		new_num++;
		g_free(to_path);
	}

	if (fp) fclose(fp);
	g_free(to_dir);
	CHDIR_RETURN_IF_FAIL(maildir);
}

void procmsg_clean_trash(void)
{
	DIR *dp;
	struct dirent *d;

	CHDIR_RETURN_IF_FAIL(maildir);
	CHDIR_RETURN_IF_FAIL(TRASH_DIR);

	if ((dp = opendir(".")) == NULL) {
		perror("opendir");
		CHDIR_RETURN_IF_FAIL(maildir);
		return;
	}

	while ((d = readdir(dp)) != NULL) {
		if (to_number(d->d_name) < 0) continue;
		debug_print(_("Unlinking message %s in trash...\n"), d->d_name);
		if (unlink(d->d_name) < 0)
			perror("unlink");
	}

	debug_print(_("done.\n"));
	closedir(dp);

	CHDIR_RETURN_IF_FAIL(maildir);
}

void procmsg_msginfo_set(MsgInfo *msginfo, const gchar *msgnum,
			 const gchar *folder, MsgFlags default_flag)
{
	gchar *file;
	struct stat s;

	memset(msginfo, 0, sizeof(MsgInfo));
	msginfo->msgnum = atoi(msgnum);
	msginfo->folder = g_strdup(folder);
	msginfo->flags  = default_flag;

	file = g_strdup_printf("%s/%s/%s", maildir, folder, msgnum);
	procheader_parse(msginfo, file);

	if (stat(file, &s) < 0) {
		perror("stat");
		msginfo->size = 0;
		msginfo->mtime = 0;
	} else {
		msginfo->size = s.st_size;
		msginfo->mtime = s.st_mtime;
	}

	g_free(file);
}

void procmsg_msginfo_free(MsgInfo *msginfo)
{
	if (msginfo == NULL) return;

	g_free(msginfo->folder);
	g_free(msginfo->to_folder);
	g_free(msginfo->fromname);

	g_free(msginfo->date);
	g_free(msginfo->from);
	g_free(msginfo->to);
	g_free(msginfo->newsgroups);
	g_free(msginfo->subject);
	g_free(msginfo->msgid);
	g_free(msginfo->inreplyto);

	g_free(msginfo);
}

static gint procmsg_cmp_msgnum(gconstpointer a, gconstpointer b)
{
	const MsgInfo *msginfo = a;
	const guint msgnum = GPOINTER_TO_UINT(b);

	if (!msginfo)
		return -1;

	return msginfo->msgnum - msgnum;
}

static gint procmsg_cmp_msgnum_for_sort(gconstpointer a, gconstpointer b)
{
	const MsgInfo *msginfo1 = a;
	const MsgInfo *msginfo2 = b;

	if (!msginfo1)
		return -1;
	if (!msginfo2)
		return -1;

	return msginfo1->msgnum - msginfo2->msgnum;
}

static gint procmsg_cmp_flag_msgnum(gconstpointer a, gconstpointer b)
{
	const FlagInfo *finfo = a;
	const guint msgnum = GPOINTER_TO_UINT(b);

	if (!finfo)
		return -1;

	return finfo->msgnum - msgnum;
}
