/* send.c */

#include <glib.h>
#include <stdio.h>
#include <string.h>

#include "main.h"
#include "send.h"
#include "socket.h"
#include "smtp.h"
#include "account.h"
#include "procmsg.h"
#include "utils.h"

static gint send_smtp_open(void);

#define SEND_EXIT_IF_ERROR(f, s) \
{ \
	if ((f) < 0) { \
		g_warning("Error occured while %s\n", s); \
		fclose(fp); \
		sock_close(smtp_sock); \
		return -1; \
	} \
}

#define SEND_EXIT_IF_NOTOK(f, s) \
{ \
	if ((f) != SM_OK) { \
		g_warning("Error occured while %s\n", s); \
		fclose(fp); \
		if (smtp_quit(smtp_sock) != SM_OK) \
			g_warning("Error occured while sending QUIT\n"); \
		sock_close(smtp_sock); \
		return -1; \
	} \
}

#define SEND_RETURN_IF_ERROR(f, s) \
{ \
	if ((f) < 0) { \
		g_warning("Error occured while %s\n", s); \
		return -1; \
	} \
}

gint send_message_smtp(GSList *to_list, const gchar *file)
{
	gint smtp_sock;
	FILE *fp;
	gchar buf[BUFFSIZE];
	GSList *cur;

	g_return_val_if_fail(to_list != NULL, -1);
	g_return_val_if_fail(file != NULL, -1);
	g_return_val_if_fail(cur_account != NULL, -1);

	if ((fp = fopen(file, "r")) == NULL) {
		perror("fopen");
		return -1;
	}

	SEND_EXIT_IF_ERROR((smtp_sock = send_smtp_open()),
			   "connecting to server");
	SEND_EXIT_IF_NOTOK(smtp_from(smtp_sock, cur_account->address),
			   "sending MAIL FROM");
	for (cur = to_list; cur != NULL; cur = cur->next)
		SEND_EXIT_IF_NOTOK(smtp_rcpt(smtp_sock, (gchar *)cur->data),
				   "sending RCPT TO");
	SEND_EXIT_IF_NOTOK(smtp_data(smtp_sock), "sending DATA");

	/* send main part */
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		strretchomp(buf);

		/* escape when a dot appears on the top */
		if (buf[0] == '.')
			SEND_EXIT_IF_ERROR(sock_write(smtp_sock, ".", 1),
					   "sending data");

		SEND_EXIT_IF_ERROR(sock_puts(smtp_sock, buf), "sending data");
	}

	SEND_EXIT_IF_NOTOK(smtp_eom(smtp_sock), "terminating data");
	SEND_EXIT_IF_NOTOK(smtp_quit(smtp_sock), "sending QUIT");

	sock_close(smtp_sock);
	fclose(fp);
	return 0;
}

static gint send_smtp_open(void)
{
	gint smtp_sock;
	gint val;

	g_return_val_if_fail(cur_account != NULL, -1);

	if ((smtp_sock = sock_connect
	     (cur_account->smtp_server,
	      cur_account->set_smtpport ? cur_account->smtpport : 25)) < 0) {
		fprintf(stderr, "Can't connect to SMTP server.\n");
		return -1;
	}

	if (smtp_ok(smtp_sock) == SM_OK) {
		if (cur_account->set_domain &&
		    cur_account->domain && *cur_account->domain)
			val = smtp_helo(smtp_sock, cur_account->domain);
		else
			val = smtp_helo(smtp_sock, get_domain_name());

		if (val == SM_OK)
			return smtp_sock;
	}

	fprintf(stderr, "Error occured while sending HELO\n");
	sock_close(smtp_sock);
	return -1;
}
