/* socket.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>

#include "socket.h"

#define BUFFSIZE	8192

gint sock_connect(const gchar *hostname, gushort port)
{
	struct hostent *hp;
	struct sockaddr_in ad;
	gint sock;
#ifndef HAVE_INET_ATON
#if HAVE_INET_ADDR
	guint32 inaddr;
#endif
#endif /* HAVE_INET_ATON */

	memset(&ad, 0, sizeof(ad));
	ad.sin_family = AF_INET;
	ad.sin_port = htons(port);

#if HAVE_INET_ATON
	if (!inet_aton(hostname, &ad.sin_addr)) {
#else
#if HAVE_INET_ADDR
	inaddr = inet_addr(hostname);
	if (inaddr != -1)
		memcpy(&ad.sin_addr, &inaddr, sizeof(inaddr));
	else {
#else
	{
#endif
#endif /* HAVE_INET_ATON */
		if ((hp = gethostbyname(hostname)) == NULL) {
			fprintf(stderr, "%s: unknown host.\n", hostname);
			errno = 0;
			return -1;
		}

		if (hp->h_length != 4 && hp->h_length != 8) {
			h_errno = errno = 0;
			fprintf(stderr, "illegal address length received for host %s\n", hostname);
			return -1;
		}

		memcpy(&ad.sin_addr, hp->h_addr, hp->h_length);
	}

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("socket");
		return -1;
	}
	if (connect(sock, (struct sockaddr *)&ad, sizeof(ad)) < 0) {
		perror("connect");
		return -1;
	}

	return sock;
}

gint sock_printf(gint sock, const gchar *format, ...)
{
	va_list args;
	gchar buf[BUFFSIZE];

	va_start(args, format);
	g_vsnprintf(buf, sizeof(buf), format, args);
	va_end(args);

	return sock_write(sock, buf, strlen(buf));
}

gint sock_write(gint sock, const gchar *buf, gint len)
{
	gint n, wrlen = 0;

	while (len) {
		n = write(sock, buf, len);
		if (n <= 0)
			return -1;
		len -= n;
		wrlen += n;
		buf += n;
	}

	return wrlen;
}

gint sock_read(gint sock, gchar *buf, gint len)
{
	gchar *newline, *bp = buf;
	gint n;

	if (--len < 1)
		return -1;
	do {
		if ((n = recv(sock, bp, len, MSG_PEEK)) <= 0)
			return -1;
		if ((newline = memchr(bp, '\n', n)) != NULL)
			n = newline - bp + 1;
		if ((n = read(sock, bp, n)) < 0)
			return -1;
		bp += n;
		len -= n;
	} while (!newline && len);

	*bp = '\0';
	return bp - buf;
}

gint sock_puts(gint sock, const gchar *buf)
{
	gint ret;

	if ((ret = sock_write(sock, buf, strlen(buf))) < 0)
		return ret;
	return sock_write(sock, "\r\n", 2);
}

/* peek at the next socket character without actually reading it */
gint sock_peek(gint sock)
{
	gint n;
	gchar ch;

	if ((n = recv(sock, &ch, 1, MSG_PEEK)) < 0)
		return -1;
	else
		return ch;
}

gint sock_close(gint sock)
{
	return close(sock);
}
