/* summaryview.c */

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkbindings.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkpixmap.h>
#include <gtk/gtkctree.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtksignal.h>
#include <gtk/gtktext.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkitemfactory.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkstyle.h>
#include <gtk/gtkarrow.h>
#include <gtk/gtkeventbox.h>
#include <gtk/gtkstatusbar.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>

#include "main.h"
#include "menu.h"
#include "mainwindow.h"
#include "folderview.h"
#include "summaryview.h"
#include "headerview.h"
#include "messageview.h"
#include "foldersel.h"
#include "procmsg.h"
#include "procheader.h"
#include "unmime.h"
#include "headerwindow.h"
#include "prefs_common.h"
#include "compose.h"
#include "utils.h"
#include "gtkutils.h"
#include "alertpanel.h"
#include "news.h"
#include "statusbar.h"
#include "filter.h"

#include "pixmaps/dir-open.xpm"
//#include "regular.xpm"
#include "pixmaps/new.xpm"
#include "pixmaps/unread.xpm"

typedef enum
{
	COL_MARK	= 0,
	COL_UNREAD	= 1,
	COL_NUMBER	= 2,
	COL_DATE	= 3,
	COL_FROM	= 4,
	COL_SUBJECT	= 5
} SummaryColumnPos;

#define N_SUMMARY_COLS	6

#define STATUSBAR_PUSH(mainwin, str) \
{ \
	gtk_statusbar_push(GTK_STATUSBAR(mainwin->statusbar), \
			   mainwin->summaryview_cid, str); \
}

#define STATUSBAR_POP(mainwin) \
{ \
	gtk_statusbar_pop(GTK_STATUSBAR(mainwin->statusbar), \
			  mainwin->summaryview_cid); \
}

static GdkFont *smallfont;

static GdkPixmap *folderxpm;
static GdkBitmap *folderxpmmask;
//static GdkPixmap *regularxpm;
//static GdkBitmap *regularxpmmask;
static GdkPixmap *newxpm;
static GdkBitmap *newxpmmask;
static GdkPixmap *unreadxpm;
static GdkBitmap *unreadxpmmask;

static void summary_free_msginfo_func	(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 gpointer		 data);
static void summary_set_marks_func	(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 gpointer		 data);
static void summary_write_cache_func	(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 gpointer		 data);

static GtkCTreeNode *summary_find_next_unread_msg
					(SummaryView		*summaryview,
					 GtkCTreeNode		*current_node);
#if 0
static GtkCTreeNode *summary_find_prev_unread_msg
					(SummaryView		*summaryview,
					 GtkCTreeNode		*current_node);
#endif

static void summary_update_status	(SummaryView		*summaryview);

/* display functions */
static void summary_status_show		(SummaryView		*summaryview);
static void summary_set_ctree_from_list	(SummaryView		*summaryview,
					 GSList			*mlist);
static void summary_set_header		(gchar			*text[],
					 MsgInfo		*msginfo);
static void summary_display_msg		(SummaryView		*summaryview,
					 GtkCTreeNode		*row);
static void summary_step		(SummaryView		*summaryview,
					 GtkScrollType		 type);
static void summary_toggle_view		(SummaryView		*summaryview);

/* message handling */
static void summary_mark_row		(SummaryView		*summaryview,
					 GtkCTreeNode		*row);
static void summary_mark		(SummaryView		*summaryview);
static void summary_mark_as_read	(SummaryView		*summaryview);
static void summary_mark_as_unread	(SummaryView		*summaryview);
static void summary_delete_row		(SummaryView		*summaryview,
					 GtkCTreeNode		*row);
static void summary_delete		(SummaryView		*summaryview);
static void summary_unmark_row		(SummaryView		*summaryview,
					 GtkCTreeNode		*row);
static void summary_unmark		(SummaryView		*summaryview);
static void summary_move_row_to		(SummaryView		*summaryview,
					 GtkCTreeNode		*row,
					 const gchar		*to_folder);
static void summary_move_to		(SummaryView		*summaryview);
static void summary_select_all		(SummaryView		*summaryview);
static void summary_unselect_all	(SummaryView		*summaryview);

static void summary_delete_duplicated_func
					(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 SummaryView		*summaryview);

static void summary_execute_move	(SummaryView		*summaryview);
static void summary_execute_move_func	(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 gpointer		 data);
static void summary_execute_delete	(SummaryView		*summaryview);
static void summary_execute_delete_func	(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 gpointer		 data);

/* thread functions */
static void summary_thread_func		(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 gpointer		 data);
static void summary_unthread_func	(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 gpointer		 data);

static void summary_assort_func		(GtkCTree		*ctree,
					 GtkCTreeNode		*node,
					 gpointer		 data);

/* callback functions */
static void summary_toggle_pressed	(GtkWidget		*eventbox,
					 GdkEventButton		*event,
					 SummaryView		*summaryview);
static void summary_button_pressed	(GtkWidget		*ctree,
					 GdkEventButton		*event,
					 SummaryView		*summaryview);
static void summary_button_released	(GtkWidget		*ctree,
					 GdkEventButton		*event,
					 SummaryView		*summaryview);
static void summary_key_pressed		(GtkWidget		*ctree,
					 GdkEventKey		*event,
					 SummaryView		*summaryview);
static void summary_selected		(GtkCTree		*ctree,
					 GtkCTreeNode		*row,
					 gint			 column,
					 SummaryView		*summaryview);
static void summary_reply_cb		(SummaryView		*summaryview,
					 guint			 action,
					 GtkWidget		*widget);

static void summary_num_clicked		(GtkWidget		*button,
					 SummaryView		*summaryview);
static void summary_date_clicked	(GtkWidget		*button,
					 SummaryView		*summaryview);
static void summary_from_clicked	(GtkWidget		*button,
					 SummaryView		*summaryview);
static void summary_subject_clicked	(GtkWidget		*button,
					 SummaryView		*summaryview);

/* custom compare functions for sorting */

static gint summary_cmp_by_num		(GtkCList		*clist,
					 gconstpointer		 ptr1,
					 gconstpointer		 ptr2);
static gint summary_cmp_by_date		(GtkCList		*clist,
					 gconstpointer		 ptr1,
					 gconstpointer		 ptr2);
static gint summary_cmp_by_from		(GtkCList		*clist,
					 gconstpointer		 ptr1,
					 gconstpointer		 ptr2);
static gint summary_cmp_by_subject	(GtkCList		*clist,
					 gconstpointer		 ptr1,
					 gconstpointer		 ptr2);

static GtkItemFactoryEntry summary_popup_entries[] =
{
	{N_("/M_ove..."),		NULL, summary_move_to,	0, NULL},
	{N_("/_Copy..."),		NULL, NULL,		0, NULL},
	{N_("/_Delete"),		NULL, summary_delete,	0, NULL},
	{N_("/_Save as..."),		NULL, NULL,		0, NULL},
	{N_("/E_xecute"),		NULL, summary_execute,	0, NULL},
	{N_("/---"),			NULL, NULL,		0, "<Separator>"},
	{N_("/_Reply"),			NULL, summary_reply_cb,	0, NULL},
	{N_("/Reply with _quotation"),	NULL, summary_reply_cb,	1, NULL},
	{N_("/_Forward"),		NULL, summary_reply_cb, 2, NULL},
	{N_("/---"),			NULL, NULL,		0, "<Separator>"},
	{N_("/_Mark"),			NULL, summary_mark,	0, NULL},
	{N_("/_Unmark"),		NULL, summary_unmark,	0, NULL},
	{N_("/Mark all"),		NULL, NULL,		0, NULL},
	{N_("/Unmark all"),		NULL, NULL,		0, NULL},
	{N_("/Move marked"),		NULL, NULL,		0, NULL},
	{N_("/Delete marked"),		NULL, NULL,		0, NULL},
	{N_("/---"),			NULL, NULL,		0, "<Separator>"},
	{N_("/Mark as unr_ead"),	NULL, summary_mark_as_unread, 0, NULL},
	{N_("/Mark as _important"),	NULL, NULL,		0, NULL},
	{N_("/Make it as _being read"),	NULL, summary_mark_as_read, 0, NULL},
	{N_("/---"),			NULL, NULL,		0, "<Separator>"},
	{N_("/Select _all"),		NULL, summary_select_all, 0, NULL},
	{N_("/U_nselect all"),		NULL, summary_unselect_all, 0, NULL}
};

SummaryView *summary_create(void)
{
	SummaryView *summaryview;
	gchar *titles[N_SUMMARY_COLS] = {_("M"), _("U")};
	GtkWidget *vbox;
	GtkWidget *scrolledwin;
	GtkWidget *ctree;
	GtkWidget *hbox;
	GtkWidget *statlabel_folder;
	GtkWidget *statlabel_select;
	GtkWidget *statlabel_msgs;
	GtkWidget *toggle_eventbox;
	GtkWidget *toggle_arrow;
	GtkWidget *popupmenu;
	GtkItemFactory *popupfactory;
	GtkBindingSet *binding_set;
	gint n_entries;
	gint i;

	debug_print(_("Creating summary view...\n"));
	summaryview = g_new0(SummaryView, 1);

	vbox = gtk_vbox_new(FALSE, 2);

	scrolledwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwin),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_widget_set_usize(scrolledwin,
			     prefs_common.mainview_width,
			     DEFAULT_SUMMARY_HEIGHT);
	gtk_box_pack_start(GTK_BOX(vbox), scrolledwin, TRUE, TRUE, 0);

	if (prefs_common.trans_hdr) {
		titles[COL_NUMBER]  = _("No.");
		titles[COL_DATE]    = _("Date");
		titles[COL_FROM]    = _("From");
		titles[COL_SUBJECT] = _("Subject");
	} else {
		titles[COL_NUMBER]  = "No.";
		titles[COL_DATE]    = "Date";
		titles[COL_FROM]    = "From";
		titles[COL_SUBJECT] = "Subject";
	}

	ctree = gtk_ctree_new_with_titles(N_SUMMARY_COLS, COL_SUBJECT, titles);
	gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(scrolledwin),
					    GTK_CLIST(ctree)->vadjustment);
	gtk_container_add(GTK_CONTAINER(scrolledwin), ctree);
	gtk_clist_set_selection_mode(GTK_CLIST(ctree), GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_justification(GTK_CLIST(ctree), COL_NUMBER,
					   GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_MARK, 10);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_UNREAD, 10);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_NUMBER, 40);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_DATE, 134);
	gtk_clist_set_column_width(GTK_CLIST(ctree), COL_FROM, 140);
	gtk_ctree_set_line_style(GTK_CTREE(ctree), GTK_CTREE_LINES_DOTTED);
#if 0
	gtk_ctree_set_line_style(GTK_CTREE(ctree), GTK_CTREE_LINES_NONE);
	gtk_ctree_set_expander_style(GTK_CTREE(ctree),
				     GTK_CTREE_EXPANDER_TRIANGLE);
#endif
	gtk_ctree_set_indent(GTK_CTREE(ctree), 18);
	//gtk_clist_set_reorderable(GTK_CLIST(ctree), TRUE);
	gtk_object_set_user_data(GTK_OBJECT(ctree), summaryview);

	/* don't let title buttons take key focus */
	for (i = 0; i < N_SUMMARY_COLS; i++)
		GTK_WIDGET_UNSET_FLAGS(GTK_CLIST(ctree)->column[i].button,
				       GTK_CAN_FOCUS);

	/* connect signal to the buttons for sorting */
	gtk_signal_connect
		(GTK_OBJECT(GTK_CLIST(ctree)->column[COL_NUMBER].button),
		 "clicked",
		 GTK_SIGNAL_FUNC(summary_num_clicked),
		 summaryview);
	gtk_signal_connect
		(GTK_OBJECT(GTK_CLIST(ctree)->column[COL_DATE].button),
		 "clicked",
		 GTK_SIGNAL_FUNC(summary_date_clicked),
		 summaryview);
	gtk_signal_connect
		(GTK_OBJECT(GTK_CLIST(ctree)->column[COL_FROM].button),
		 "clicked",
		 GTK_SIGNAL_FUNC(summary_from_clicked),
		 summaryview);
	gtk_signal_connect
		(GTK_OBJECT(GTK_CLIST(ctree)->column[COL_SUBJECT].button),
		 "clicked",
		 GTK_SIGNAL_FUNC(summary_subject_clicked),
		 summaryview);

	/* create status label */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	statlabel_folder = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(hbox), statlabel_folder, FALSE, FALSE, 2);
	statlabel_select = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(hbox), statlabel_select, FALSE, FALSE, 16);

	/* toggle view button */
	toggle_eventbox = gtk_event_box_new();
	gtk_box_pack_end(GTK_BOX(hbox), toggle_eventbox, FALSE, FALSE, 4);
	toggle_arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(toggle_eventbox), toggle_arrow);

	statlabel_msgs = gtk_label_new("");
	gtk_box_pack_end(GTK_BOX(hbox), statlabel_msgs, FALSE, FALSE, 4);

	/* create popup menu */
	n_entries = sizeof(summary_popup_entries) /
		sizeof(summary_popup_entries[0]);
	popupmenu = menu_create_items(summary_popup_entries, n_entries,
				      "<SummaryView>", &popupfactory,
				      summaryview);

	menu_set_sensitive(popupfactory, "/Copy...",       FALSE);
	menu_set_sensitive(popupfactory, "/Save as...",    FALSE);
	menu_set_sensitive(popupfactory, "/Mark as important", FALSE);
	menu_set_sensitive(popupfactory, "/Mark all",      FALSE);
	menu_set_sensitive(popupfactory, "/Unmark all",    FALSE);
	menu_set_sensitive(popupfactory, "/Move marked",   FALSE);
	menu_set_sensitive(popupfactory, "/Delete marked", FALSE);

	/* bind keys */
	binding_set = gtk_binding_set_by_class
		(GTK_CLIST_CLASS(((GtkObject *)(ctree))->klass));
#if 0
	gtk_binding_entry_add_signal(binding_set, GDK_n, 0,
				     "scroll_vertical", 2,
				     GTK_TYPE_ENUM, GTK_SCROLL_STEP_FORWARD,
				     GTK_TYPE_FLOAT, 0.0);
	gtk_binding_entry_add_signal(binding_set, GDK_p, 0,
				     "scroll_vertical", 2,
				     GTK_TYPE_ENUM, GTK_SCROLL_STEP_BACKWARD,
				     GTK_TYPE_FLOAT, 0.0);
#endif /* 0 */
	gtk_binding_entry_clear(binding_set, GDK_space, 0);

	/* connect signals */
	gtk_signal_connect(GTK_OBJECT(ctree), "tree_select_row",
			   GTK_SIGNAL_FUNC(summary_selected), summaryview);
	gtk_signal_connect(GTK_OBJECT(ctree), "button_press_event",
			   GTK_SIGNAL_FUNC(summary_button_pressed),
			   summaryview);
	gtk_signal_connect(GTK_OBJECT(ctree), "button_release_event",
			   GTK_SIGNAL_FUNC(summary_button_released),
			   summaryview);
	gtk_signal_connect(GTK_OBJECT(ctree), "key_press_event",
			   GTK_SIGNAL_FUNC(summary_key_pressed), summaryview);

	gtk_signal_connect(GTK_OBJECT(toggle_eventbox), "button_press_event",
			   GTK_SIGNAL_FUNC(summary_toggle_pressed),
			   summaryview);

	summaryview->vbox = vbox;
	summaryview->scrolledwin = scrolledwin;
	summaryview->ctree = ctree;
	summaryview->hbox = hbox;
	summaryview->statlabel_folder = statlabel_folder;
	summaryview->statlabel_select = statlabel_select;
	summaryview->statlabel_msgs = statlabel_msgs;
	summaryview->toggle_eventbox = toggle_eventbox;
	summaryview->toggle_arrow = toggle_arrow;
	summaryview->popupmenu = popupmenu;
	summaryview->popupfactory = popupfactory;
	summaryview->msg_is_toggled_on = TRUE;

	gtk_widget_show_all(vbox);

	return summaryview;
}

void summary_init(SummaryView *summaryview)
{
	GtkStyle *style;
	GtkWidget *pixmap;

	//PIXMAP_CREATE(summaryview->ctree, regularxpm, regularxpmmask,
	//		regular_xpm);
	PIXMAP_CREATE(summaryview->ctree, newxpm, newxpmmask, new_xpm);
	PIXMAP_CREATE(summaryview->ctree, unreadxpm, unreadxpmmask, unread_xpm);
	PIXMAP_CREATE(summaryview->hbox, folderxpm, folderxpmmask,
		      DIRECTORY_OPEN_XPM);

	if (!smallfont)
		smallfont = gdk_fontset_load(SMALL_FONT);

	style = gtk_style_copy(gtk_widget_get_style
				(summaryview->statlabel_folder));
	style->font = smallfont;
	gtk_widget_set_style(summaryview->statlabel_folder, style);
	gtk_widget_set_style(summaryview->statlabel_select, style);
	gtk_widget_set_style(summaryview->statlabel_msgs, style);

	pixmap = gtk_pixmap_new(folderxpm, folderxpmmask);
	gtk_box_pack_start(GTK_BOX(summaryview->hbox), pixmap, FALSE, FALSE, 4);
	gtk_box_reorder_child(GTK_BOX(summaryview->hbox), pixmap, 0);
	gtk_widget_show(pixmap);
}

void summary_show(SummaryView *summaryview, FolderInfo *finfo,
		  gboolean update_cache)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GtkCTreeNode *node;
	GSList *mlist;
	gchar *buf;

	STATUSBAR_POP(summaryview->mainwin);

	/* write to cache file before clearing summary */
	summary_write_cache(summaryview);

	gtk_clist_freeze(GTK_CLIST(ctree));

	summary_clear_list(summaryview);

	if (finfo && finfo->type == F_NEWSGROUP) {
		gchar *server;

		summaryview->mode = SUMMARY_NEWS;
		server = g_dirname(finfo->path);
		debug_print("server = %s, group = %s\n",
			    server, g_basename(finfo->path));
		buf = g_strdup_printf(_("Connecting to %s ..."), server);
		STATUSBAR_PUSH(summaryview->mainwin, buf);
		g_free(buf);
		main_window_cursor_wait(summaryview->mainwin);
		summaryview->news_session = news_session_get(server);
		STATUSBAR_POP(summaryview->mainwin);
		g_free(server);
	}

	if (!finfo || (summaryview->mode == SUMMARY_MAIL &&
	     (change_dir(maildir) < 0 || !is_dir_exist(finfo->path)))) {
		debug_print(_("empty folder\n\n"));
		summary_clear_all(summaryview);
		gtk_clist_thaw(GTK_CLIST(ctree));
		return;
	}

	summaryview->cur_folder = g_strdup(finfo->path);

	gtk_signal_disconnect_by_data(GTK_OBJECT(ctree), summaryview);

	buf = g_strdup_printf(_("Scanning folder (%s)..."), finfo->path);
	debug_print("%s\n", buf);
	STATUSBAR_PUSH(summaryview->mainwin, buf);
	g_free(buf);

	main_window_cursor_wait(summaryview->mainwin);

	/* get messages list */
	if (summaryview->mode == SUMMARY_MAIL)
		mlist = procmsg_get_msgs(finfo->path, !update_cache);
	else
		mlist = news_get_article_info(summaryview->news_session,
					      finfo->path, !update_cache);

	STATUSBAR_POP(summaryview->mainwin);

	/* set ctree and hash table from the msginfo list
	   creating thread, and count the number of messages */
	summary_set_ctree_from_list(summaryview, mlist);

	slist_remove_all(mlist);

	summary_thread_build(summaryview);

	/* write to cache file */
	summary_write_cache(summaryview);

	gtk_signal_connect(GTK_OBJECT(ctree), "tree_select_row",
			   GTK_SIGNAL_FUNC(summary_selected), summaryview);
	gtk_signal_connect(GTK_OBJECT(ctree), "button_press_event",
			   GTK_SIGNAL_FUNC(summary_button_pressed),
			   summaryview);
	gtk_signal_connect(GTK_OBJECT(ctree), "button_release_event",
			   GTK_SIGNAL_FUNC(summary_button_released),
			   summaryview);
	gtk_signal_connect(GTK_OBJECT(ctree), "key_press_event",
			   GTK_SIGNAL_FUNC(summary_key_pressed), summaryview);

	gtk_clist_thaw(GTK_CLIST(ctree));

	/* select first unread message */
	node = summary_find_next_unread_msg(summaryview, NULL);
	if (node == NULL && GTK_CLIST(ctree)->row_list != NULL)
		node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list_end);
	if (node) {
		gtk_ctree_node_moveto(ctree, node, COL_MARK, 0.5, 0);
		gtk_widget_grab_focus(GTK_WIDGET(ctree));
		summaryview->display_msg = TRUE;
		gtk_ctree_select(ctree, node);
		gtkut_ctree_set_focus_row(ctree, node);
	}

	summary_status_show(summaryview);

	if (summaryview->selected) {
		gtk_widget_set_sensitive(summaryview->mainwin->reply_btn,
					 TRUE);
		gtk_widget_set_sensitive(summaryview->mainwin->quote_btn,
					 TRUE);
		gtk_widget_set_sensitive(summaryview->mainwin->fwd_btn,
					 TRUE);
	} else {
		gtk_widget_set_sensitive(summaryview->mainwin->reply_btn,
					 FALSE);
		gtk_widget_set_sensitive(summaryview->mainwin->quote_btn,
					 FALSE);
		gtk_widget_set_sensitive(summaryview->mainwin->fwd_btn,
					 FALSE);
	}

	debug_print("\n");
	STATUSBAR_PUSH(summaryview->mainwin, _("done."));

	main_window_cursor_normal(summaryview->mainwin);
}

void summary_clear_list(SummaryView *summaryview)
{
	gtk_clist_freeze(GTK_CLIST(summaryview->ctree));

	gtk_ctree_pre_recursive(GTK_CTREE(summaryview->ctree),
				NULL, summary_free_msginfo_func, NULL);

	summaryview->mode = SUMMARY_MAIL;

	g_free(summaryview->cur_folder);
	summaryview->cur_folder = NULL;
	summaryview->news_session = NULL;

	summaryview->selected = NULL;
	summaryview->displayed = NULL;
	summaryview->newmsgs = summaryview->unread = 0;
	summaryview->messages = summaryview->total_size = 0;
	summaryview->deleted = summaryview->moved = 0;
	if (summaryview->msgid_table) {
		g_hash_table_destroy(summaryview->msgid_table);
		summaryview->msgid_table = NULL;
	}
	summaryview->mlist = NULL;
	if (summaryview->folder_table) {
		g_hash_table_destroy(summaryview->folder_table);
		summaryview->folder_table = NULL;
	}

	gtk_clist_clear(GTK_CLIST(summaryview->ctree));

	gtk_clist_thaw(GTK_CLIST(summaryview->ctree));
}

void summary_clear_all(SummaryView *summaryview)
{
	summary_clear_list(summaryview);

	gtk_widget_set_sensitive(summaryview->mainwin->reply_btn, FALSE);
	gtk_widget_set_sensitive(summaryview->mainwin->quote_btn, FALSE);
	gtk_widget_set_sensitive(summaryview->mainwin->fwd_btn, FALSE);

	summary_status_show(summaryview);
}

static GtkCTreeNode *summary_find_next_unread_msg(SummaryView *summaryview,
						  GtkCTreeNode *current_node)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GtkCTreeNode *node;
	MsgInfo *msginfo;

	if (current_node)
		node = current_node;
		//node = GTK_CTREE_NODE_NEXT(current_node);
	else
		node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);

	for (; node != NULL; node = GTK_CTREE_NODE_NEXT(node)) {
		msginfo = gtk_ctree_node_get_row_data(ctree, node);
		if (MSG_IS_UNREAD(msginfo->flags)) break;
	}

	return node;
}

#if 0
static GtkCTreeNode *summary_find_prev_unread_msg(SummaryView *summaryview,
						  GtkCTreeNode *current_node)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GtkCTreeNode *node;
	MsgInfo *msginfo;

	if (current_node)
		node = current_node;
		//node = GTK_CTREE_NODE_PREV(current_node);
	else
		node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list_end);

	for (; node != NULL; node = GTK_CTREE_NODE_PREV(node)) {
		msginfo = gtk_ctree_node_get_row_data(ctree, node);
		if (MSG_IS_UNREAD(msginfo->flags)) break;
	}

	return node;
}
#endif

void summary_attract_by_subject(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GtkCList *clist = GTK_CLIST(ctree);
	GtkCTreeNode *src_node;
	GtkCTreeNode *dst_node, *sibling;
	GtkCTreeNode *tmp;
	MsgInfo *src_msginfo, *dst_msginfo;

	debug_print(_("Attracting messages by subject..."));
	STATUSBAR_PUSH(summaryview->mainwin,
		       _("Attracting messages by subject..."));

	main_window_cursor_wait(summaryview->mainwin);
	gtk_clist_freeze(clist);

	for (src_node = GTK_CTREE_NODE(clist->row_list);
	     src_node != NULL;
	     src_node = tmp) {
		tmp = GTK_CTREE_ROW(src_node)->sibling;
		src_msginfo = GTKUT_CTREE_NODE_GET_ROW_DATA(src_node);
		if (!src_msginfo) continue;
		if (!src_msginfo->subject) continue;

		/* find attracting node */
		for (dst_node = GTK_CTREE_NODE_PREV(src_node);
		     dst_node != NULL;
		     dst_node = GTK_CTREE_NODE_PREV(dst_node)) {
			/* ignore child node */
			if (GTK_CTREE_ROW(dst_node)->level > 1) continue;

			dst_msginfo = GTKUT_CTREE_NODE_GET_ROW_DATA(dst_node);
			if (!dst_msginfo->subject) continue;

			/* if the time difference is more than 30 days,
			   skip the comparison */
			if (ABS(src_msginfo->date_t - dst_msginfo->date_t)
			    > 2592000)
				continue;

			if (!subject_compare(src_msginfo->subject,
					     dst_msginfo->subject)) {
				sibling = GTK_CTREE_ROW(dst_node)->sibling;
				if (src_node != sibling)
					gtk_ctree_move(ctree, src_node,
						       NULL, sibling);
				break;
			}
		}
	}

	gtk_clist_thaw(clist);

	debug_print(_("done.\n"));
	STATUSBAR_POP(summaryview->mainwin);

	main_window_cursor_normal(summaryview->mainwin);
}

static void summary_free_msginfo_func(GtkCTree *ctree, GtkCTreeNode *node,
				      gpointer data)
{
	MsgInfo *msginfo = gtk_ctree_node_get_row_data(ctree, node);

	if (msginfo)
		procmsg_msginfo_free(msginfo);
}

static void summary_set_marks_func(GtkCTree *ctree, GtkCTreeNode *node,
				   gpointer data)
{
	SummaryView *summaryview = data;
	MsgInfo *msginfo;

	msginfo = gtk_ctree_node_get_row_data(ctree, node);

	if (MSG_IS_NEW(msginfo->flags)) {
		gtk_ctree_node_set_pixmap(ctree, node, COL_UNREAD,
					  newxpm, newxpmmask);
		summaryview->newmsgs++;
		if (MSG_IS_UNREAD(msginfo->flags))
			summaryview->unread++;
	} else if (MSG_IS_UNREAD(msginfo->flags)) {
		gtk_ctree_node_set_pixmap(ctree, node, COL_UNREAD,
					  unreadxpm, unreadxpmmask);
		summaryview->unread++;
	}

	summaryview->messages++;
	summaryview->total_size += msginfo->size;
}

static void summary_update_status(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GtkCTreeNode *node;
	MsgInfo *msginfo;

	summaryview->newmsgs = summaryview->unread =
	summaryview->messages = summaryview->total_size =
	summaryview->deleted = summaryview->moved = 0;

	for (node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);
	     node != NULL; node = GTK_CTREE_NODE_NEXT(node)) {
		msginfo = GTKUT_CTREE_NODE_GET_ROW_DATA(node);

		if (MSG_IS_NEW(msginfo->flags))
			summaryview->newmsgs++;
		if (MSG_IS_UNREAD(msginfo->flags))
			summaryview->unread++;
		if (MSG_IS_DELETED(msginfo->flags))
			summaryview->deleted++;
		if (MSG_IS_MOVE(msginfo->flags))
			summaryview->moved++;
		summaryview->messages++;
		summaryview->total_size += msginfo->size;
	}
}

static void summary_status_show(SummaryView *summaryview)
{
	gchar *buf1, *buf2, *buf3;
	GList *rowlist, *cur;
	guint n_selected = 0;
	off_t sel_size = 0;
	MsgInfo *msginfo;

	if (!summaryview->cur_folder) {
		gtk_label_set(GTK_LABEL(summaryview->statlabel_folder), "");
		gtk_label_set(GTK_LABEL(summaryview->statlabel_select), "");
		gtk_label_set(GTK_LABEL(summaryview->statlabel_msgs),   "");
		return;
	}

	rowlist = GTK_CLIST(summaryview->ctree)->selection;
	for (cur = rowlist; cur != NULL; cur = cur->next) {
		msginfo = gtk_ctree_node_get_row_data
			(GTK_CTREE(summaryview->ctree),
			 GTK_CTREE_NODE(cur->data));
		sel_size += msginfo->size;
		n_selected++;
	}

	gtk_label_set(GTK_LABEL(summaryview->statlabel_folder),
		      summaryview->cur_folder);

	if (summaryview->deleted)
		buf1 = g_strdup_printf(_("%d deleted"), summaryview->deleted);
	else
		buf1 = g_strdup("");
	if (summaryview->moved)
		buf2 = g_strdup_printf(_("%d moved"), summaryview->moved);
	else
		buf2 = g_strdup("");

	buf3 = g_strdup_printf("%s%s%s%s",
			       summaryview->deleted || summaryview->moved
						? "    " : "", buf1,
			       summaryview->deleted && summaryview->moved
						? _(", ") : "", buf2);

	g_free(buf1);
	g_free(buf2);
	if (n_selected)
		buf2 = g_strdup_printf(" (%s)", to_human_readable(sel_size));
	else
		buf2 = g_strdup("");
	buf1 = g_strdup_printf("%s%s%s%s",
			      n_selected ? itos(n_selected) : "",
			      n_selected ? _(" item(s) selected") : "",
			      buf2, buf3);
	gtk_label_set(GTK_LABEL(summaryview->statlabel_select), buf1);
	g_free(buf1);
	g_free(buf2);
	g_free(buf3);

	if (summaryview->mode == SUMMARY_MAIL) {
		buf1 = g_strdup_printf(_("%d new, %d unread, %d total (%s)"),
				       summaryview->newmsgs,
				       summaryview->unread,
				       summaryview->messages,
				       to_human_readable(summaryview->total_size));
	} else {
		buf1 = g_strdup_printf(_("%s new, %d unread, %d total"),
				       summaryview->newmsgs,
				       summaryview->unread,
				       summaryview->messages);
	}
	gtk_label_set(GTK_LABEL(summaryview->statlabel_msgs), buf1);
	g_free(buf1);

	folderview_update_msg_num(summaryview->folderview,
				  summaryview->folderview->opened,
				  summaryview->newmsgs,
				  summaryview->unread,
				  summaryview->messages);
}

void summary_sort(SummaryView *summaryview, SummarySortType type)
{
	GtkCListCompareFunc cmp_func;

	switch (type) {
	case SORT_BY_NUMBER:
		cmp_func = (GtkCListCompareFunc)summary_cmp_by_num;
		break;
	case SORT_BY_DATE:
		cmp_func = (GtkCListCompareFunc)summary_cmp_by_date;
		break;
	case SORT_BY_FROM:
		cmp_func = (GtkCListCompareFunc)summary_cmp_by_from;
		break;
	case SORT_BY_SUBJECT:
		cmp_func = (GtkCListCompareFunc)summary_cmp_by_subject;
		break;
	default:
		return;
	}

	debug_print(_("Sorting summary..."));
	STATUSBAR_PUSH(summaryview->mainwin, _("Sorting summary..."));

	main_window_cursor_wait(summaryview->mainwin);

	gtk_clist_set_compare_func(GTK_CLIST(summaryview->ctree), cmp_func);
	gtk_ctree_sort_node(GTK_CTREE(summaryview->ctree), NULL);

	debug_print(_("done.\n"));
	STATUSBAR_POP(summaryview->mainwin);

	main_window_cursor_normal(summaryview->mainwin);
}

static void summary_set_ctree_from_list(SummaryView *summaryview,
					GSList *mlist)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	MsgInfo *msginfo;
	GtkCTreeNode *node, *parent;
	gchar *text[N_SUMMARY_COLS];
	GHashTable *msgid_table;

	if (!mlist) return;

	debug_print(_("\tSetting summary from message data..."));
	STATUSBAR_PUSH(summaryview->mainwin,
		       _("Setting summary from message data..."));
	GTK_EVENTS_FLUSH();

	msgid_table = g_hash_table_new(g_str_hash, g_str_equal);

	if (/*prefs_common.thread_on*/ 1) {
		for (; mlist != NULL; mlist = mlist->next) {
			msginfo = (MsgInfo *)mlist->data;
			parent = NULL;

			summary_set_header(text, msginfo);

			/* search parent node for threading */
			if (msginfo->inreplyto && *msginfo->inreplyto)
				parent = g_hash_table_lookup
					(msgid_table, msginfo->inreplyto);

			node = gtk_ctree_insert_node
				(ctree, parent, NULL, text, 2,
				 NULL, NULL, NULL, NULL, FALSE, TRUE);
			GTKUT_CTREE_NODE_SET_ROW_DATA(node, msginfo);
			summary_set_marks_func(ctree, node, summaryview);

			/* preserve previous node if the message is
			   duplicated */
			if (msginfo->msgid && *msginfo->msgid &&
			    g_hash_table_lookup(msgid_table, msginfo->msgid)
			    == NULL)
				g_hash_table_insert(msgid_table,
						    msginfo->msgid, node);
		}
	} else {
		for (; mlist != NULL; mlist = mlist->next) {
			msginfo = (MsgInfo *)mlist->data;

			summary_set_header(text, msginfo);

			node = gtk_ctree_insert_node
				(ctree, NULL, NULL, text, 2,
				 NULL, NULL, NULL, NULL, FALSE, TRUE);
			GTKUT_CTREE_NODE_SET_ROW_DATA(node, msginfo);
			summary_set_marks_func(ctree, node, summaryview);

			if (msginfo->msgid && *msginfo->msgid &&
			    g_hash_table_lookup(msgid_table, msginfo->msgid)
			    == NULL)
				g_hash_table_insert(msgid_table,
						    msginfo->msgid, node);
		}
	}

	summaryview->msgid_table = msgid_table;

	debug_print(_("done.\n"));
	STATUSBAR_POP(summaryview->mainwin);
	debug_print("\tmsgid hash table size = %d\n",
		    g_hash_table_size(msgid_table));
}

struct wcachefp
{
	FILE *cache_fp;
	FILE *mark_fp;
};

#define MAKE_DIR_IF_NOT_EXIST(dir) \
{ \
	if (!is_dir_exist(dir)) { \
		if (mkdir(dir, S_IRWXU) < 0) { \
			fprintf(stderr, "%s: ", dir); \
			perror("mkdir"); \
			g_free(dir); \
			return -1; \
		} \
		if (chmod(path, S_IRWXU) < 0) { \
			fprintf(stderr, "%s: ", dir); \
			perror("chmod"); \
		} \
	} \
}

gint summary_write_cache(SummaryView *summaryview)
{
	struct wcachefp fps;
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	gint ver = CACHE_VERSION;
	gchar *buf;
	gchar *cachefile, *markfile;

	if (!summaryview->cur_folder) return -1;

	if (summaryview->mode == SUMMARY_NEWS) {
		gchar *path, *server;

		server = g_dirname(summaryview->cur_folder);
		path = g_strconcat(get_news_cache_dir(), G_DIR_SEPARATOR_S,
				   server, NULL);
		g_free(server);
		MAKE_DIR_IF_NOT_EXIST(path);
		g_free(path);

		path = g_strconcat(get_news_cache_dir(), G_DIR_SEPARATOR_S,
				   summaryview->cur_folder, NULL);
		MAKE_DIR_IF_NOT_EXIST(path);
		cachefile = g_strconcat(path, G_DIR_SEPARATOR_S, CACHE_FILE,
					NULL);
		g_free(path);
	} else
		cachefile = g_strconcat(maildir, G_DIR_SEPARATOR_S,
					summaryview->cur_folder,
					G_DIR_SEPARATOR_S,
					CACHE_FILE, NULL);

	if ((fps.cache_fp = fopen(cachefile, "w")) == NULL) {
		fprintf(stderr, "%s: ", cachefile);
		perror("fopen");
		g_free(cachefile);
		return -1;
	}
	g_free(cachefile);

	markfile = g_strconcat(summaryview->mode == SUMMARY_MAIL ? maildir :
			       get_news_cache_dir(), G_DIR_SEPARATOR_S,
			       summaryview->cur_folder, G_DIR_SEPARATOR_S,
			       MARK_FILE, NULL);

	if ((fps.mark_fp = fopen(markfile, "w")) == NULL) {
		fprintf(stderr, "%s: ", markfile);
		perror("fopen");
		fclose(fps.cache_fp);
		g_free(markfile);
		return -1;
	}
	g_free(markfile);

	buf = g_strdup_printf(_("Writing summary cache (%s)..."),
			      summaryview->cur_folder);
	debug_print(buf);
	STATUSBAR_PUSH(summaryview->mainwin, buf);
	g_free(buf);

	WRITE_CACHE_DATA_INT(ver, fps.cache_fp);
	ver = MARK_VERSION;
	WRITE_CACHE_DATA_INT(ver, fps.mark_fp);

	gtk_ctree_pre_recursive(ctree, NULL, summary_write_cache_func, &fps);

	fclose(fps.cache_fp);
	fclose(fps.mark_fp);

	debug_print(_("done.\n"));
	STATUSBAR_POP(summaryview->mainwin);

	return 0;
}

static void summary_write_cache_func(GtkCTree *ctree, GtkCTreeNode *node,
				     gpointer data)
{
	struct wcachefp *fps = data;
	MsgInfo *msginfo = gtk_ctree_node_get_row_data(ctree, node);

	if (msginfo == NULL) return;

	procmsg_write_cache(msginfo, fps->cache_fp);
	procmsg_write_flags(msginfo, fps->mark_fp);
}

static void summary_set_header(gchar *text[], MsgInfo *msginfo)
{
	static gchar date_modified[80];

	text[COL_MARK]   = NULL;
	text[COL_UNREAD] = NULL;
	text[COL_NUMBER] = itos(msginfo->msgnum);
	if (msginfo->date_t) {
		procheader_date_get_localtime(date_modified,
					      sizeof(date_modified),
					      msginfo->date_t);
		text[COL_DATE] = date_modified;
	} else if (msginfo->date)
		text[COL_DATE] = msginfo->date;
	else
		text[COL_DATE] = _("(No Date)");

	text[COL_FROM] = msginfo->fromname ? msginfo->fromname :
		_("(No From)");

	text[COL_SUBJECT] = msginfo->subject ? msginfo->subject :
		_("(No Subject)");
}

static void summary_display_msg(SummaryView *summaryview, GtkCTreeNode *row)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	MsgInfo *msginfo;
	gchar *filename;

	if (summaryview->displayed == row) return;
	g_return_if_fail(row != NULL);

	STATUSBAR_POP(summaryview->mainwin);

	summaryview->displayed = row;

	msginfo = gtk_ctree_node_get_row_data(ctree, row);

	if (summaryview->mode == SUMMARY_MAIL) {
		filename = g_strconcat(summaryview->cur_folder,
				       G_DIR_SEPARATOR_S,
				       itos(msginfo->msgnum), NULL);
	} else if (summaryview->mode == SUMMARY_NEWS) {
		gint ok;

		filename = g_strconcat(get_news_cache_dir(), G_DIR_SEPARATOR_S,
				       summaryview->cur_folder,
				       G_DIR_SEPARATOR_S,
				       itos(msginfo->msgnum), NULL);
		if (!is_file_exist(filename)) {
			if (!summaryview->news_session) {
				g_warning(_("news session is not established\n"));
				g_free(filename);
				return;
			}

			debug_print(_("getting article %d...\n"),
				    msginfo->msgnum);
			ok = news_get_article(summaryview->news_session,
					      msginfo->msgnum,
					      filename);
			statusbar_pop_all();
			if (ok < 0) {
				g_warning(_("can't read article %d\n"),
					  msginfo->msgnum);
				g_free(filename);
				return;
			}
			debug_print(_("done.\n"));
		} else
			debug_print(_("article %d already has been cached.\n"),
				    msginfo->msgnum);
	} else
		return;

	if (MSG_IS_NEW(msginfo->flags))
		summaryview->newmsgs--;
	if (MSG_IS_UNREAD(msginfo->flags))
		summaryview->unread--;
	if (MSG_IS_NEW(msginfo->flags) || MSG_IS_UNREAD(msginfo->flags)) {
		MSG_UNSET_FLAGS(msginfo->flags, MSG_NEW | MSG_UNREAD);
		gtk_ctree_node_set_text(ctree, row, COL_UNREAD, NULL);
		gtk_clist_thaw(GTK_CLIST(ctree));
		summary_status_show(summaryview);
	}

	headerview_show(summaryview->headerview, msginfo);
	messageview_show(summaryview->messageview, filename);

	if (!summaryview->msg_is_toggled_on) {
		summary_toggle_view(summaryview);
		GTK_EVENTS_FLUSH();
	}

	gtk_ctree_node_moveto(ctree, row, COL_MARK, 0.5, 0);

	if (GTK_WIDGET_VISIBLE(summaryview->headerwin->window))
		header_window_show(summaryview->headerwin, filename);

	g_free(filename);
}

static void summary_step(SummaryView *summaryview, GtkScrollType type)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);

	gtk_signal_emit_by_name(GTK_OBJECT(ctree), "scroll_vertical",
				type, 0.0);

	if (summaryview->msg_is_toggled_on)
		summary_display_msg(summaryview, summaryview->selected);
}

static void summary_toggle_view(SummaryView *summaryview)
{
	MainWindow *mainwin = summaryview->mainwin;
	union CompositeWin *cwin = &mainwin->win;
	GtkWidget *vpaned = NULL;
	GtkWidget *container = NULL;

	switch (mainwin->type) {
	case SEPARATE_NONE:
		vpaned = cwin->sep_none.vpaned;
		container = cwin->sep_none.hpaned;
		break;
	case SEPARATE_FOLDER:
		vpaned = cwin->sep_folder.vpaned;
		container = mainwin->vbox_body;
		break;
	case SEPARATE_MESSAGE:
	case SEPARATE_BOTH:
		return;
	}

	if (vpaned->parent != NULL) {
		summaryview->msg_is_toggled_on = FALSE;
		summaryview->displayed = NULL;
		gtk_widget_ref(vpaned);
		gtk_container_remove(GTK_CONTAINER(container), vpaned);
		gtk_widget_reparent(summaryview->vbox, container);
		gtk_arrow_set(GTK_ARROW(summaryview->toggle_arrow),
			      GTK_ARROW_UP, GTK_SHADOW_OUT);
	} else {
		summaryview->msg_is_toggled_on = TRUE;
		gtk_widget_reparent(summaryview->vbox, vpaned);
		gtk_container_add(GTK_CONTAINER(container), vpaned);
		gtk_widget_unref(vpaned);
		gtk_arrow_set(GTK_ARROW(summaryview->toggle_arrow),
			      GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	}

	gtk_widget_grab_focus(summaryview->ctree);
}

static void summary_mark_row(SummaryView *summaryview, GtkCTreeNode *row)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	MsgInfo *msginfo;

	msginfo = gtk_ctree_node_get_row_data(ctree, row);
	g_free(msginfo->to_folder);
	msginfo->to_folder = NULL;
	if (MSG_IS_DELETED(msginfo->flags))
		summaryview->deleted--;
	if (MSG_IS_MOVE(msginfo->flags))
		summaryview->moved--;
	MSG_UNSET_FLAGS(msginfo->flags, MSG_DELETED | MSG_MOVE | MSG_COPY);
	MSG_SET_FLAGS(msginfo->flags, MSG_MARKED);
	gtk_ctree_node_set_text(ctree, row, COL_MARK, "*");
	gtk_ctree_node_set_foreground
		(ctree, row, &summaryview->color_marked);
	debug_print(_("Message %d is marked\n"), msginfo->msgnum);
}

static void summary_mark(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GList *cur;

	for (cur = GTK_CLIST(ctree)->selection; cur != NULL; cur = cur->next)
		summary_mark_row(summaryview, (GtkCTreeNode *)cur->data);

	summary_step(summaryview, GTK_SCROLL_STEP_FORWARD);
}

static void summary_mark_as_read(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GList *cur;
	GtkCTreeNode *row;
	MsgInfo *msginfo;

	for (cur = GTK_CLIST(ctree)->selection; cur != NULL;
	     cur = cur->next) {
		row = cur->data;
		msginfo = gtk_ctree_node_get_row_data(ctree, row);
		if (MSG_IS_NEW(msginfo->flags))
			summaryview->newmsgs--;
		if (MSG_IS_UNREAD(msginfo->flags))
			summaryview->unread--;
		if (MSG_IS_NEW(msginfo->flags) ||
		    MSG_IS_UNREAD(msginfo->flags)) {
			MSG_UNSET_FLAGS(msginfo->flags, MSG_NEW | MSG_UNREAD);
			gtk_ctree_node_set_text(ctree, row, COL_UNREAD, NULL);
			debug_print(_("Message %d is marked as being read\n"),
				    msginfo->msgnum);
		}
	}

	summary_status_show(summaryview);
}

static void summary_mark_as_unread(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GList *cur;
	GtkCTreeNode *row;
	MsgInfo *msginfo;

	for (cur = GTK_CLIST(ctree)->selection; cur != NULL;
	     cur = cur->next) {
		row = cur->data;
		msginfo = gtk_ctree_node_get_row_data(ctree, row);
		if (MSG_IS_DELETED(msginfo->flags)) {
			g_free(msginfo->to_folder);
			msginfo->to_folder = NULL;
			MSG_UNSET_FLAGS(msginfo->flags, MSG_DELETED);
			summaryview->deleted--;
		}
		if (!MSG_IS_UNREAD(msginfo->flags)) {
			MSG_SET_FLAGS(msginfo->flags, MSG_UNREAD);
			gtk_ctree_node_set_pixmap(ctree, row, COL_UNREAD,
						  unreadxpm, unreadxpmmask);
			summaryview->unread++;
			gtk_ctree_node_set_foreground
				(ctree, row, &summaryview->color_normal);
			debug_print(_("Message %d is marked as unread\n"),
				    msginfo->msgnum);
		}
	}

	summary_status_show(summaryview);
}

static void summary_delete_row(SummaryView *summaryview, GtkCTreeNode *row)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	MsgInfo *msginfo;

	msginfo = gtk_ctree_node_get_row_data(ctree, row);

	if (MSG_IS_DELETED(msginfo->flags)) return;

	g_free(msginfo->to_folder);
	msginfo->to_folder = NULL;
	if (MSG_IS_MOVE(msginfo->flags))
		summaryview->moved--;
	MSG_UNSET_FLAGS(msginfo->flags,
			MSG_MARKED |
			MSG_MOVE |
			MSG_COPY);
	MSG_SET_FLAGS(msginfo->flags, MSG_DELETED);
	summaryview->deleted++;

	gtk_ctree_node_set_text(ctree, row, COL_MARK, _("D"));
	gtk_ctree_node_set_foreground(ctree, row, &summaryview->color_dim);
	debug_print(_("Message %s/%d is set to delete\n"),
		    msginfo->folder, msginfo->msgnum);
}

static void summary_delete(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GList *cur;

	/* if current folder is trash, don't delete */
	if (!strcmp2(summaryview->cur_folder, TRASH_DIR)) {
		alertpanel(_("Can't delete"), _("Current folder is Trash."),
			   _("OK"), NULL, NULL);
		return;
	}

	for (cur = GTK_CLIST(ctree)->selection; cur != NULL; cur = cur->next)
		summary_delete_row(summaryview, GTK_CTREE_NODE(cur->data));

	summary_status_show(summaryview);
	summary_step(summaryview, GTK_SCROLL_STEP_FORWARD);
}

void summary_delete_duplicated(SummaryView *summaryview)
{
	main_window_cursor_wait(summaryview->mainwin);
	debug_print(_("Deleting duplicated messages..."));
	STATUSBAR_PUSH(summaryview->mainwin,
		       _("Deleting duplicated messages..."));
	GTK_EVENTS_FLUSH();

	gtk_ctree_pre_recursive(GTK_CTREE(summaryview->ctree), NULL,
				GTK_CTREE_FUNC(summary_delete_duplicated_func),
				summaryview);

	summary_status_show(summaryview);

	debug_print(_("done.\n"));
	STATUSBAR_POP(summaryview->mainwin);
	main_window_cursor_normal(summaryview->mainwin);
}

static void summary_delete_duplicated_func(GtkCTree *ctree, GtkCTreeNode *node,
					   SummaryView *summaryview)
{
	GtkCTreeNode *found;
	MsgInfo *msginfo = GTK_CTREE_ROW(node)->row.data;

	if (!msginfo->msgid || !*msginfo->msgid) return;

	found = g_hash_table_lookup(summaryview->msgid_table, msginfo->msgid);

	if (found && found != node)
		summary_delete_row(summaryview, node);
}

static void summary_unmark_row(SummaryView *summaryview, GtkCTreeNode *row)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	MsgInfo *msginfo;

	msginfo = gtk_ctree_node_get_row_data(ctree, row);
	g_free(msginfo->to_folder);
	msginfo->to_folder = NULL;
	if (MSG_IS_DELETED(msginfo->flags))
		summaryview->deleted--;
	if (MSG_IS_MOVE(msginfo->flags))
		summaryview->moved--;
	MSG_UNSET_FLAGS(msginfo->flags,
			MSG_MARKED |
			MSG_DELETED |
			MSG_IMPORTANT |
			MSG_MOVE |
			MSG_COPY);
	gtk_ctree_node_set_text(ctree, row, COL_MARK, NULL);
	gtk_ctree_node_set_foreground
		(ctree, row, &summaryview->color_normal);
	debug_print(_("Message %s/%d is unmarked\n"),
		    msginfo->folder, msginfo->msgnum);
}

static void summary_unmark(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GList *cur;

	for (cur = GTK_CLIST(ctree)->selection; cur != NULL;
	     cur = cur->next)
		summary_unmark_row(summaryview, GTK_CTREE_NODE(cur->data));

	summary_status_show(summaryview);
}

static void summary_move_row_to(SummaryView *summaryview, GtkCTreeNode *row,
				const gchar *to_folder)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	MsgInfo *msginfo;

	msginfo = gtk_ctree_node_get_row_data(ctree, row);
	g_free(msginfo->to_folder);
	msginfo->to_folder = g_strdup(to_folder);
	if (MSG_IS_DELETED(msginfo->flags))
		summaryview->deleted--;
	MSG_UNSET_FLAGS(msginfo->flags,
			MSG_MARKED | MSG_DELETED | MSG_COPY);
	if (!MSG_IS_MOVE(msginfo->flags)) {
		MSG_SET_FLAGS(msginfo->flags, MSG_MOVE);
		summaryview->moved++;
	}
	gtk_ctree_node_set_text(ctree, row, COL_MARK, _("o"));
	gtk_ctree_node_set_foreground
		(ctree, row, &summaryview->color_marked);
	debug_print(_("Message %d is set to move to %s\n"),
		    msginfo->msgnum, to_folder);
}

static void summary_move_to(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	gchar *to_folder;
	GList *cur;

	to_folder = foldersel_folder_sel(NULL);

	if (!to_folder) return;

	for (cur = GTK_CLIST(ctree)->selection; cur != NULL; cur = cur->next)
		summary_move_row_to
			(summaryview, (GtkCTreeNode *)cur->data, to_folder);

	summary_status_show(summaryview);
	summary_step(summaryview, GTK_SCROLL_STEP_FORWARD);
}

static void summary_select_all(SummaryView *summaryview)
{
	gtk_clist_select_all(GTK_CLIST(summaryview->ctree));
}

static void summary_unselect_all(SummaryView *summaryview)
{
	gtk_clist_unselect_all(GTK_CLIST(summaryview->ctree));
}

void summary_execute(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GtkCList *clist = GTK_CLIST(summaryview->ctree);
	GtkCTreeNode *node, *next;

	gtk_clist_freeze(clist);

	summary_unthread(summaryview);

	summary_execute_move(summaryview);
	summary_execute_delete(summaryview);

	node = GTK_CTREE_NODE(clist->row_list);
	while (node != NULL) {
		next = GTK_CTREE_NODE_NEXT(node);
		if (gtk_ctree_node_get_row_data(ctree, node) == NULL)
			gtk_ctree_remove_node(ctree, node);
		node = next;
	}

	summary_thread_build(summaryview);

	summaryview->selected = clist->selection ?
		GTK_CTREE_NODE(clist->selection->data) : NULL;

	summary_update_status(summaryview);
	summary_status_show(summaryview);

	summary_write_cache(summaryview);

	gtk_clist_thaw(clist);
}

static void summary_execute_move(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GSList *cur;

	summaryview->folder_table = g_hash_table_new(g_str_hash, g_str_equal);

	/* search moving messages and execute */
	gtk_ctree_pre_recursive(ctree, NULL, summary_execute_move_func,
				summaryview);

	if (summaryview->mlist) {
		procmsg_move_messages(summaryview->cur_folder,
				      summaryview->mlist);

		/* update folder tree */
		g_hash_table_foreach(summaryview->folder_table,
				     folderview_scan_foreach_func,
				     summaryview->folderview);

		for (cur = summaryview->mlist; cur != NULL; cur = cur->next)
			procmsg_msginfo_free((MsgInfo *)cur->data);

		slist_remove_all(summaryview->mlist);
		summaryview->mlist = NULL;
	}

	g_hash_table_destroy(summaryview->folder_table);
	summaryview->folder_table = NULL;
}

static void summary_execute_move_func(GtkCTree *ctree, GtkCTreeNode *node,
				      gpointer data)
{
	SummaryView *summaryview = data;
	MsgInfo *msginfo;

	msginfo = GTKUT_CTREE_NODE_GET_ROW_DATA(node);

	if (msginfo && MSG_IS_MOVE(msginfo->flags) && msginfo->to_folder) {
		gint val;

		if ((val = GPOINTER_TO_INT(g_hash_table_lookup
					   (summaryview->folder_table,
					    msginfo->to_folder))))
			g_hash_table_insert(summaryview->folder_table,
					    msginfo->to_folder,
					    GINT_TO_POINTER(val + 1));
		else
			g_hash_table_insert(summaryview->folder_table,
					    g_strdup(msginfo->to_folder),
					    GINT_TO_POINTER(1));

		summaryview->mlist =
			g_slist_append(summaryview->mlist, msginfo);
		gtk_ctree_node_set_row_data(ctree, node, NULL);

		if (msginfo->msgid && *msginfo->msgid &&
		    node == g_hash_table_lookup(summaryview->msgid_table,
						msginfo->msgid))
			g_hash_table_remove(summaryview->msgid_table,
					    msginfo->msgid);
	}
}

static void summary_execute_delete(SummaryView *summaryview)
{
	GtkCTree *ctree = GTK_CTREE(summaryview->ctree);
	GSList *cur;

	/* search deleting messages and execute */
	gtk_ctree_pre_recursive
		(ctree, NULL, summary_execute_delete_func, summaryview);

	if (!summaryview->mlist) return;

	procmsg_move_messages_with_dest(TRASH_DIR, summaryview->cur_folder,
					summaryview->mlist);

	folderview_scan_folder(summaryview->folderview, TRASH_DIR);

	for (cur = summaryview->mlist; cur != NULL; cur = cur->next)
		procmsg_msginfo_free((MsgInfo *)cur->data);

	slist_remove_all(summaryview->mlist);
	summaryview->mlist = NULL;
}

static void summary_execute_delete_func(GtkCTree *ctree, GtkCTreeNode *node,
					gpointer data)
{
	SummaryView *summaryview = data;
	MsgInfo *msginfo;

	msginfo = GTKUT_CTREE_NODE_GET_ROW_DATA(node);

	if (msginfo && MSG_IS_DELETED(msginfo->flags)) {
		summaryview->mlist =
			g_slist_append(summaryview->mlist, msginfo);
		gtk_ctree_node_set_row_data(ctree, node, NULL);

		if (msginfo->msgid && *msginfo->msgid &&
		    node == g_hash_table_lookup(summaryview->msgid_table,
						msginfo->msgid))
			g_hash_table_remove(summaryview->msgid_table,
					    msginfo->msgid);
	}
}

/* thread functions */

void summary_thread_build(SummaryView *summaryview)
{
	debug_print(_("Building threads..."));
	STATUSBAR_PUSH(summaryview->mainwin, _("Building threads..."));
	GTK_EVENTS_FLUSH();

	gtk_clist_freeze(GTK_CLIST(summaryview->ctree));

	gtk_ctree_pre_recursive_to_depth
		(GTK_CTREE(summaryview->ctree), NULL, 1,
		 GTK_CTREE_FUNC(summary_thread_func),
		 summaryview->msgid_table);

	gtk_clist_thaw(GTK_CLIST(summaryview->ctree));

	debug_print(_("done.\n"));
	STATUSBAR_POP(summaryview->mainwin);
	GTK_EVENTS_FLUSH();
}

void summary_unthread(SummaryView *summaryview)
{
	debug_print(_("Unthreading..."));
	STATUSBAR_PUSH(summaryview->mainwin, _("Unthreading..."));
	GTK_EVENTS_FLUSH();

	gtk_clist_freeze(GTK_CLIST(summaryview->ctree));

	gtk_ctree_pre_recursive(GTK_CTREE(summaryview->ctree), NULL,
				GTK_CTREE_FUNC(summary_unthread_func), NULL);

	gtk_clist_thaw(GTK_CLIST(summaryview->ctree));

	debug_print(_("done.\n"));
	STATUSBAR_POP(summaryview->mainwin);
	GTK_EVENTS_FLUSH();
}

static void summary_thread_func(GtkCTree *ctree, GtkCTreeNode *node,
				gpointer data)
{
	GtkCTreeNode *parent;
	MsgInfo *msginfo = GTK_CTREE_ROW(node)->row.data;
	GHashTable *msgid_table = (GHashTable *)data;

	if (!msginfo->inreplyto || !*msginfo->inreplyto) return;

	parent = g_hash_table_lookup(msgid_table, msginfo->inreplyto);

	if (parent && parent != node) {
		gtk_ctree_move(ctree, node, parent, NULL);
		gtk_ctree_expand(ctree, parent);
	}
}

static void summary_unthread_func(GtkCTree *ctree, GtkCTreeNode *node,
				  gpointer data)
{
	GtkCTreeNode *parent = GTK_CTREE_ROW(node)->parent;

	if (!parent || GTK_CTREE_ROW(node)->level != 2) return;

	if (GTK_CTREE_ROW(node)->sibling)
		summary_unthread_func(ctree, GTK_CTREE_ROW(node)->sibling,
				      NULL);

	gtk_ctree_move(ctree, node, NULL, GTK_CTREE_ROW(parent)->sibling);
}

void summary_assort(SummaryView *summaryview)
{
	if (!prefs_common.fltlist) return;

	debug_print(_("assorting..."));
	STATUSBAR_PUSH(summaryview->mainwin, _("Assorting..."));
	main_window_cursor_wait(summaryview->mainwin);

	gtk_clist_freeze(GTK_CLIST(summaryview->ctree));

	gtk_ctree_pre_recursive(GTK_CTREE(summaryview->ctree), NULL,
				GTK_CTREE_FUNC(summary_assort_func),
				summaryview);

	gtk_clist_thaw(GTK_CLIST(summaryview->ctree));

	summary_status_show(summaryview);

	debug_print(_("done.\n"));
	STATUSBAR_POP(summaryview->mainwin);
	main_window_cursor_normal(summaryview->mainwin);
}

static void summary_assort_func(GtkCTree *ctree, GtkCTreeNode *node,
				gpointer data)
{
	MsgInfo *msginfo = GTKUT_CTREE_NODE_GET_ROW_DATA(node);
	SummaryView *summaryview = data;
	gchar *file;
	gchar *dest;

	file = g_strdup_printf("%s%c%d", msginfo->folder, G_DIR_SEPARATOR,
			       msginfo->msgnum);
	dest = filter_get_dest_folder(prefs_common.fltlist, file);
	g_free(file);

	if (dest)
		summary_move_row_to(summaryview, node, dest);
}

/* callback functions */

static void summary_toggle_pressed(GtkWidget *eventbox, GdkEventButton *event,
				   SummaryView *summaryview)
{
	if (!event)
		return;

	if (!summaryview->msg_is_toggled_on && summaryview->selected)
		summary_display_msg(summaryview, summaryview->selected);
	else
		summary_toggle_view(summaryview);
}

static void summary_button_pressed(GtkWidget *ctree, GdkEventButton *event,
				   SummaryView *summaryview)
{
	GtkCList *clist = GTK_CLIST(ctree);
	gint row, column;

	if (!event) return;

	if (event->button == 3) {
		/* right clicked */
		if (clist->selection && !clist->selection->next) {
			gtk_clist_unselect_all(clist);
			gtk_clist_get_selection_info(clist,
						     event->x, event->y,
						     &row, &column);
			gtk_clist_select_row(clist, row, column);
			gtkut_clist_set_focus_row(clist, row);
		}
		gtk_menu_popup(GTK_MENU(summaryview->popupmenu), NULL, NULL,
			       NULL, NULL, event->button, event->time);
	} else if (event->button == 2) {
		/* center clicked */
		gtk_clist_unselect_all(clist);
		gtk_clist_get_selection_info(clist, event->x, event->y,
					     &row, &column);
		gtk_clist_select_row(clist, row, column);
		gtkut_clist_set_focus_row(clist, row);
		if (clist->selection)
			summary_display_msg
				(summaryview,
				 GTK_CTREE_NODE(clist->selection->data));
	} else if (event->button == 1)
		summaryview->display_msg = TRUE;
}

static void summary_button_released(GtkWidget *ctree, GdkEventButton *event,
				    SummaryView *summaryview)
{
}

void summary_pass_key_press_event(SummaryView *summaryview, GdkEventKey *event)
{
	summary_key_pressed(summaryview->ctree, event, summaryview);
}

static void summary_key_pressed(GtkWidget *widget, GdkEventKey *event,
				SummaryView *summaryview)
{
	GtkText *text = GTK_TEXT(summaryview->messageview->text);
	GtkCTree *ctree = GTK_CTREE(widget);
	GtkCList *clist = GTK_CLIST(widget);
	GtkCTreeNode *node;
	gfloat upper;
	gchar *to_folder;

	if (!event)
		return;
	if (!summaryview->selected) {
		node = gtk_ctree_node_nth(ctree, 0);
		if (node)
			gtk_ctree_select(ctree, node);
		else
			return;
	}

	switch (event->keyval) {
	case GDK_space:		/* Page down or go to the next */
		if (summaryview->displayed != summaryview->selected) {
			summary_display_msg(summaryview,
					    summaryview->selected);
			break;
		}

		upper = text->vadj->upper - text->vadj->page_size;
		if (text->vadj->value < upper) {
			text->vadj->value += text->vadj->page_increment;
			text->vadj->value = MIN(text->vadj->value, upper);
			gtk_signal_emit_by_name(GTK_OBJECT(text->vadj),
						"value_changed");
			break;
		}

		/* if can't page down any more, go to next unread message */
		node = summary_find_next_unread_msg(summaryview,
						    summaryview->selected);
		if (node) {
			gtk_clist_unselect_all(clist);
			gtk_ctree_select(ctree, node);
			gtkut_ctree_set_focus_row(ctree, node);
			summary_display_msg(summaryview, summaryview->selected);
		} else {
			AlertValue val;

			val = alertpanel(_("No unread message"),
					 _("No unread message found.\n"
					 "Go to next folder?"),
					 _("Yes"), _("No"), NULL);
			if (val == G_ALERTDEFAULT) {
				gtk_signal_emit_stop_by_name
					(GTK_OBJECT(ctree), "key_press_event");
				folderview_select_next_unread
					(summaryview->folderview);
			}
		}
		break;
	case GDK_n:		/* Next */
	case GDK_N:
		if ((event->state & (GDK_MOD1_MASK|GDK_CONTROL_MASK)) != 0)
			break;
		summary_step(summaryview, GTK_SCROLL_STEP_FORWARD);
		break;
	case GDK_BackSpace:	/* Page up */
	case GDK_Delete:
		if (text->vadj->value > 0.0) {
			text->vadj->value -= text->vadj->page_increment;
			text->vadj->value = MAX(text->vadj->value, 0.0);
			gtk_signal_emit_by_name(GTK_OBJECT(text->vadj),
						"value_changed");
			break;
		}
		break;
	case GDK_p:		/* Prev */
	case GDK_P:
		if ((event->state & (GDK_MOD1_MASK|GDK_CONTROL_MASK)) != 0)
			break;
		summary_step(summaryview, GTK_SCROLL_STEP_BACKWARD);
		break;
	case GDK_v:
	case GDK_V:		/* Show/hide message view */
		if ((event->state & (GDK_MOD1_MASK|GDK_CONTROL_MASK)) != 0)
			break;
		if (!summaryview->msg_is_toggled_on && summaryview->selected)
			summary_display_msg(summaryview,
					    summaryview->selected);
		else
			summary_toggle_view(summaryview);
		break;
	case GDK_Return:	/* Scroll up/down one line */
		if (summaryview->displayed != summaryview->selected) {
			summary_display_msg(summaryview,
					    summaryview->selected);
			break;
		}

		if ((event->state & GDK_MOD1_MASK) == 0) {
			upper = text->vadj->upper - text->vadj->page_size;
			if (text->vadj->value < upper) {
				text->vadj->value +=
					text->vadj->step_increment * 4;
				text->vadj->value =
					MIN(text->vadj->value, upper);
				gtk_signal_emit_by_name(GTK_OBJECT(text->vadj),
							"value_changed");
			}
		} else {	/* Alt + Return */
			if (text->vadj->value > 0.0) {
				text->vadj->value -=
					text->vadj->step_increment * 4;
				text->vadj->value =
					MAX(text->vadj->value, 0.0);
				gtk_signal_emit_by_name(GTK_OBJECT(text->vadj),
							"value_changed");
			}
		}
		break;
	case GDK_asterisk:	/* Mark */
		summary_mark(summaryview);
		break;
	case GDK_exclam:	/* Mark as unread */
		summary_mark_as_unread(summaryview);
		break;
	case GDK_d:	/* Delete */
	case GDK_D:
		if ((event->state & (GDK_MOD1_MASK|GDK_CONTROL_MASK)) != 0)
			break;
		summary_delete(summaryview);
		break;
	case GDK_u:	/* Unmark */
	case GDK_U:
		if ((event->state & (GDK_MOD1_MASK|GDK_CONTROL_MASK)) != 0)
			break;
		summary_unmark(summaryview);
		break;
	case GDK_o:	/* Move */
	case GDK_O:
		if ((event->state & (GDK_MOD1_MASK|GDK_CONTROL_MASK)) != 0)
			break;
		summary_move_to(summaryview);
		break;
	case GDK_g:	/* Go */
	case GDK_G:
		if ((event->state & (GDK_MOD1_MASK|GDK_CONTROL_MASK)) != 0)
			break;
		gtk_signal_emit_stop_by_name(GTK_OBJECT(ctree),
					     "key_press_event");

		to_folder = foldersel_folder_sel(NULL);
		if (to_folder) {
			debug_print(_("Go to %s\n"), to_folder);
			folderview_select(summaryview->folderview, to_folder);
		}
		break;
	case GDK_x:	/* Execute */
	case GDK_X:
		if ((event->state & (GDK_MOD1_MASK|GDK_CONTROL_MASK)) != 0)
			break;
		summary_execute(summaryview);
		break;
	default:
	}
}

static void summary_selected(GtkCTree *ctree, GtkCTreeNode *row,
			     gint column, SummaryView *summaryview)
{
	summary_status_show(summaryview);

	if (summaryview->selected == row ||
	    (GTK_CLIST(ctree)->selection &&
	     GTK_CLIST(ctree)->selection->next)) {
		summaryview->display_msg = FALSE;
		return;
	}

	summaryview->selected = row;

	if (!prefs_common.emulate_emacs && summaryview->msg_is_toggled_on &&
	    summaryview->display_msg)
		summary_display_msg(summaryview, row);

	summaryview->display_msg = FALSE;
}

static void summary_reply_cb(SummaryView *summaryview, guint action,
			     GtkWidget *widget)
{
	MsgInfo *msginfo;

	msginfo = gtk_ctree_node_get_row_data(GTK_CTREE(summaryview->ctree),
					      summaryview->selected);
	if (!msginfo) return;

	switch (action) {
	case 0:
		compose_reply(msginfo);
		break;
	case 1:
		compose_quote(msginfo);
		break;
	case 2:
		compose_forward(msginfo);
		break;
	default:
		compose_reply(msginfo);
	}
}

static void summary_num_clicked(GtkWidget *button, SummaryView *summaryview)
{
	summary_sort(summaryview, SORT_BY_NUMBER);
}

static void summary_date_clicked(GtkWidget *button, SummaryView *summaryview)
{
	summary_sort(summaryview, SORT_BY_DATE);
}

static void summary_from_clicked(GtkWidget *button, SummaryView *summaryview)
{
	summary_sort(summaryview, SORT_BY_FROM);
}

static void summary_subject_clicked(GtkWidget *button,
				    SummaryView *summaryview)
{
	summary_sort(summaryview, SORT_BY_SUBJECT);
}

/* custom compare functions for sorting */

static gint summary_cmp_by_num(GtkCList *clist,
			       gconstpointer ptr1, gconstpointer ptr2)
{
	MsgInfo *msginfo1 = ((GtkCListRow *)ptr1)->data;
	MsgInfo *msginfo2 = ((GtkCListRow *)ptr2)->data;

	return msginfo1->msgnum - msginfo2->msgnum;
}

static gint summary_cmp_by_date(GtkCList *clist,
			       gconstpointer ptr1, gconstpointer ptr2)
{
	MsgInfo *msginfo1 = ((GtkCListRow *)ptr1)->data;
	MsgInfo *msginfo2 = ((GtkCListRow *)ptr2)->data;

	return msginfo1->date_t - msginfo2->date_t;
}

static gint summary_cmp_by_from(GtkCList *clist,
			       gconstpointer ptr1, gconstpointer ptr2)
{
	MsgInfo *msginfo1 = ((GtkCListRow *)ptr1)->data;
	MsgInfo *msginfo2 = ((GtkCListRow *)ptr2)->data;

	if (!msginfo1->fromname)
		return (msginfo2->fromname != NULL);
	if (!msginfo2->fromname)
		return -1;

	return strcasecmp(msginfo1->fromname, msginfo2->fromname);
}

static gint summary_cmp_by_subject(GtkCList *clist,
			       gconstpointer ptr1, gconstpointer ptr2)
{
	MsgInfo *msginfo1 = ((GtkCListRow *)ptr1)->data;
	MsgInfo *msginfo2 = ((GtkCListRow *)ptr2)->data;

	if (!msginfo1->subject)
		return (msginfo2->subject != NULL);
	if (!msginfo2->subject)
		return -1;

	return strcasecmp(msginfo1->subject, msginfo2->subject);
}
