/* utils.c */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#undef USE_ICONV

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#ifdef USE_ICONV
#  include <iconv.h>
#endif /* USE_ICONV */

#include <wchar.h>
#include <wctype.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdarg.h>
#include <kcc.h>
#include <sys/types.h>
#include <dirent.h>

#include "intl.h"
#include "main.h"
#include "utils.h"
#include "statusbar.h"
#include "logwindow.h"

#define BUFFSIZE	8192

#define iskanji(c) \
	(((c) & 0xff) >= 0xa1 && ((c) & 0xff) <= 0xfe)
#define iseucss(c) \
	(((c) & 0xff) == 0x8e || ((c) & 0xff) == 0x8f)
#define isunprintablekanji(c) \
	(((c) & 0xff) >= 0xa9 && ((c) & 0xff) <= 0xaf)

extern gboolean debug_mode;

void list_remove_all(GList *list)
{
	while (list != NULL)
		list = g_list_remove(list, list->data);
}

void slist_remove_all(GSList *list)
{
	while (list != NULL)
		list = g_slist_remove(list, list->data);
}

void conv_jistoeuc(gchar *outbuf, gint outlen, const gchar *inbuf)
{
#ifdef USE_ICONV
	static iconv_t cd;
	size_t inlen;

	if (!cd)
		cd = iconv_open("EUC-JP", OUT_CHARSET);

	inlen = strlen(inbuf) + 1;
	memset(outbuf, 0, inlen);
	iconv(cd, &inbuf, &inlen, &outbuf, &outlen);
#else
	KCC_filter(outbuf, "EUC", (gchar *)inbuf, "JISBB", 0, 0, 0);
#endif /* USE_ICONV */
}

void conv_euctojis(gchar *outbuf, gint outlen, const gchar *inbuf)
{
#ifdef USE_ICONV
	static iconv_t cd;
	size_t inlen;

	if (!cd)
		cd = iconv_open(OUT_CHARSET, "EUC-JP");

	inlen = strlen(inbuf) + 1;
	memset(outbuf, 0, MIN(inlen * 3, outlen));
	iconv(cd, &inbuf, &inlen, &outbuf, &outlen);
#else
	size_t inlen, len;

	inlen = strlen(inbuf);
	if (iskanji(inbuf[inlen - 1]) || iseucss(inbuf[inlen - 1])) {
		/* if tail end of the string is not ended with ascii,
		   add dummy return code. */
		gchar *tmpin, *tmpout;

		/* length of original string + '\n' + '\0' */
		tmpin = g_malloc(inlen + 2);
		if (tmpin == NULL) {
			g_warning(_("can't allocate memory\n"));
			KCC_filter(outbuf, "JISBB", (gchar *)inbuf, "EUC",
				   0, 0, 0);
			return;
		}
		strcpy(tmpin, inbuf);
		tmpin[inlen] = '\n';
		tmpin[inlen + 1] = '\0';

		tmpout = g_malloc(outlen + 1);
		if (tmpout == NULL) {
			g_warning(_("can't allocate memory\n"));
			g_free(tmpin);
			KCC_filter(outbuf, "JISBB", (gchar *)inbuf, "EUC",
				   0, 0, 0);
			return;
		}

		KCC_filter(tmpout, "JISBB", tmpin, "EUC", 0, 0, 0);
		len = strlen(tmpout);
		if (tmpout[len - 1] == '\n')
			tmpout[len - 1] = '\0';
		strncpy(outbuf, tmpout, outlen);
		outbuf[outlen - 1] = '\0';

		g_free(tmpin);
		g_free(tmpout);
	} else
		KCC_filter(outbuf, "JISBB", (gchar *)inbuf, "EUC", 0, 0, 0);
#endif /* USE_ICONV */
}

void conv_anytoeuc(gchar *outbuf, gint outlen, const gchar *inbuf)
{
	KCC_filter(outbuf, "EUC", (gchar *)inbuf, "AUTO", 0, 0, 0);
}

void conv_anytojis(gchar *outbuf, gint outlen, const gchar *inbuf)
{
	KCC_filter(outbuf, "JISBB", (gchar *)inbuf, "AUTO", 0, 0, 0);
}

#define SUBST_CHAR	'_'

void conv_unreadable(gchar *str)
{
	register gchar *p = str;

	while (*p != '\0') {
		if (isascii(*p))
			/* printable 7 bit code */
			p++;
		else if (iskanji(*p)) {
			if (iskanji(*(p + 1)) && !isunprintablekanji(*p))
				/* printable euc-jp code */
				p += 2;
			else {
				/* substitute unprintable code */
				*p++ = SUBST_CHAR;
				if (*p != '\0') {
					if (isascii(*p))
						p++;
					else
						*p++ = SUBST_CHAR;
				}
			}
		} else if (iseucss(*p)) {
			if ((*(p + 1) & 0x80) != 0)
				/* euc-jp hankaku kana */
				p += 2;
			else
				*p++ = SUBST_CHAR;
		} else
			/* substitute unprintable 1 byte code */
			*p++ = SUBST_CHAR;
	}
}

gint to_number(const gchar *nstr)
{
	register const gchar *p;

	if (*nstr == '\0') return -1;

	for (p = nstr; *p != '\0'; p++)
		if (!isdigit(*p)) return -1;

	return atoi(nstr);
}

/* convert integer into string */
gchar *itos(gint n)
{
	static gchar nstr[11];

	g_snprintf(nstr, 11, "%d", n);
	return nstr;
}

gchar *to_human_readable(off_t size)
{
	static gchar str[9];
	gint count;
	guint32 div = 1;

	for (count = 0; count < 3; count++) {
		if (size / div < 1024)
			break;
		else
			div *= 1024;
	}

	switch (count) {
	case 0: g_snprintf(str, sizeof(str), "%dB",    (gint)size);   break;
	case 1: g_snprintf(str, sizeof(str), "%.1fKB", (gfloat)size / div);
		break;
	case 2: g_snprintf(str, sizeof(str), "%.1fMB", (gfloat)size / div);
		break;
	default:
		g_snprintf(str, sizeof(str), "%.1fGB", (gfloat)size / div);
		break;
	}

	return str;
}

/* strcmp with NULL-checking */
gint strcmp2(const gchar *s1, const gchar *s2)
{
	if (s1 == NULL || s2 == NULL)
		return -1;
	else
		return strcmp(s1, s2);
}

/* compare paths */
gint path_cmp(const gchar *s1, const gchar *s2)
{
	gint len1, len2;

	if (s1 == NULL || s2 == NULL) return -1;
	if (*s1 == '\0' || *s2 == '\0') return -1;

	len1 = strlen(s1);
	len2 = strlen(s2);

	if (s1[len1 - 1] == G_DIR_SEPARATOR) len1 -= 1;
	if (s2[len2 - 1] == G_DIR_SEPARATOR) len2 -= 1;

	return strncmp(s1, s2, MAX(len1, len2));
}

/* remove trailing return code */
gchar *strretchomp(gchar *str)
{
	register gchar *s;

	if (!*str) return str;

	for (s = str + strlen(str) - 1;
	     s >= str && (*s == '\n' || *s == '\r');
	     s--)
		*s = '\0';

	return str;
}

/* Similar to `strstr' but this function ignores the case of both strings.  */
gchar *strcasestr(const gchar *haystack, const gchar *needle)
{
	register size_t haystack_len, needle_len;

	haystack_len = strlen(haystack);
	needle_len   = strlen(needle);

	if (haystack_len < needle_len || needle_len == 0)
		return NULL;

	while (haystack_len >= needle_len) {
		if (!strncasecmp(haystack, needle, needle_len))
			return (gchar *)haystack;
		else {
			haystack++;
			haystack_len--;
		}
	}

	return NULL;
}

/* Duplicate S, returning an identical malloc'd string. */
wchar_t *wcsdup(const wchar_t *s)
{
	wchar_t *new_str;

	if (s) {
		new_str = g_new(wchar_t, wcslen(s) + 1);
		wcscpy(new_str, s);
	} else
		new_str = NULL;

	return new_str;
}

/* Duplicate no more than N wide-characters of S,
   returning an identical malloc'd string. */
wchar_t *wcsndup(const wchar_t *s, size_t n)
{
	wchar_t *new_str;

	if (s) {
		new_str = g_new(wchar_t, n + 1);
		wcsncpy(new_str, s, n);
		new_str[n] = (wchar_t)0;
	} else
		new_str = NULL;

	return new_str;
}

wchar_t *strdup_mbstowcs(const gchar *s)
{
	wchar_t *new_str;

	if (s) {
		new_str = g_new(wchar_t, strlen(s) + 1);
		mbstowcs(new_str, s, strlen(s) + 1);
	} else
		new_str = NULL;

	return new_str;
}

gchar *strdup_wcstombs(const wchar_t *s)
{
	gchar *new_str;

	if (s) {
		new_str = g_new(gchar, wcslen(s) * MB_CUR_MAX + 1);
		wcstombs(new_str, s, wcslen(s) * MB_CUR_MAX + 1);
		//new_str = g_realloc(new_str, strlen(new_str) + 1);
	} else
		new_str = NULL;

	return new_str;
}

/* Compare S1 and S2, ignoring case.  */
gint wcsncasecmp(const wchar_t *s1, const wchar_t *s2, size_t n)
{
	wint_t c1;
	wint_t c2;

	while (n--) {
		c1 = towlower(*s1++);
		c2 = towlower(*s2++);
		if (c1 != c2)
			return c1 - c2;
		else if (c1 == 0 && c2 == 0)
			break;
	}

	return 0;
}

/* Find the first occurrence of NEEDLE in HAYSTACK, ignoring case.  */
wchar_t *wcscasestr(const wchar_t *haystack, const wchar_t *needle)
{
	register size_t haystack_len, needle_len;

	haystack_len = wcslen(haystack);
	needle_len   = wcslen(needle);

	if (haystack_len < needle_len || needle_len == 0)
		return NULL;

	while (haystack_len >= needle_len) {
		if (!wcsncasecmp(haystack, needle, needle_len))
			return (wchar_t *)haystack;
		else {
			haystack++;
			haystack_len--;
		}
	}

	return NULL;
}

/* Examine if next block is multi-byte string */
gboolean is_next_mbs(const wchar_t *s)
{
	gint mbl;
	const wchar_t *wp;
	gchar tmp[sizeof(wchar_t)];

	/* skip head space */
	for (wp = s; *wp != (wchar_t)0 && iswspace(*wp); wp++)
		;
	for (; *wp != (wchar_t)0 && !iswspace(*wp); wp++) {
		mbl = wctomb(tmp, *wp);
		if (mbl > 1)
			return TRUE;
	}

	return FALSE;
}

wchar_t *find_wspace(const wchar_t *s)
{
	const wchar_t *wp;

	for (wp = s; *wp != (wchar_t)0 && iswspace(*wp); wp++)
		;
	for (; *wp != (wchar_t)0; wp++) {
		if (iswspace(*wp))
			return (wchar_t *)wp;
	}

	return NULL;
}

/* compare subjects */
gint subject_compare(const gchar *s1, const gchar *s2)
{
	gint retval;
	gchar *str1, *str2;

	if (!s1 || !s2) return -1;
	if (!*s1 || !*s2) return -1;

	str1 = g_strdup(s1);
	str2 = g_strdup(s2);

	trim_subject(str1);
	trim_subject(str2);

	if (!*str1 || !*str2) {
		g_free(str1);
		g_free(str2);
		return -1;
	}

	retval = strcmp(str1, str2);
	//if (retval == 0)
	//	g_print("\ns1 = %s\ns2 = %s\n"
	//		"str1 = %s\nstr2 = %s\nmatched.\n",
	//		s1, s2, str1, str2);

	g_free(str1);
	g_free(str2);

	return retval;
}

void trim_subject(gchar *str)
{
	gchar *srcp;

	eliminate_parenthesis(str, '[', ']');
	eliminate_parenthesis(str, '(', ')');
	g_strstrip(str);

	while (!strncasecmp(str, "Re:", 3)) {
		srcp = str + 3;
		while (isspace(*srcp)) srcp++;
		memmove(str, srcp, strlen(srcp) + 1);
	}
}

void eliminate_parenthesis(gchar *str, gchar op, gchar cl)
{
	register gchar *srcp, *destp;
	gint in_brace;

	srcp = destp = str;

	while ((destp = strchr(destp, op))) {
		in_brace = 1;
		srcp = destp + 1;
		while (*srcp) {
			if (*srcp == op)
				in_brace++;
			else if (*srcp == cl)
				in_brace--;
			srcp++;
			if (in_brace == 0)
				break;
		}
		while (isspace(*srcp)) srcp++;
		memmove(destp, srcp, strlen(srcp) + 1);
	}
}

void extract_parenthesis(gchar *str, gchar op, gchar cl)
{
	register gchar *srcp, *destp;
	gint in_brace;

	srcp = destp = str;

	while ((srcp = strchr(destp, op))) {
		if (destp > str)
			*destp++ = ' ';
		memmove(destp, srcp + 1, strlen(srcp));
		in_brace = 1;
		while(*destp) {
			if (*destp == op)
				in_brace++;
			else if (*destp == cl)
				in_brace--;

			if (in_brace == 0)
				break;

			destp++;
		}
	}
	*destp = '\0';
}

void eliminate_quote(gchar *str, gchar quote_chr)
{
	register gchar *srcp, *destp;

	srcp = destp = str;

	while ((destp = strchr(destp, quote_chr))) {
		if ((srcp = strchr(destp + 1, quote_chr))) {
			srcp++;
			while (isspace(*srcp)) srcp++;
			memmove(destp, srcp, strlen(srcp) + 1);
		} else {
			*destp = '\0';
			break;
		}
	}
}

void extract_quote(gchar *str, gchar quote_chr)
{
	register gchar *p;

	if ((str = strchr(str, quote_chr))) {
		if ((p = strchr(str + 1, quote_chr))) {
			*p = '\0';
			memmove(str, str + 1, p - str);
		}
	}
}

void remove_return(gchar *str)
{
	register gchar *p = str;

	while (*p) {
		if (*p == '\n' || *p == '\r')
			memmove(p, p + 1, strlen(p));
		p++;
	}
}

void remove_space(gchar *str)
{
	register gchar *p = str;
	register gint spc;

	while (*p) {
		spc = 0;
		while (isspace(*(p + spc)))
			spc++;
		if (spc)
			memmove(p, p + spc, strlen(p + spc) + 1);
		p++;
	}
}

void subst_char(gchar *str, gchar orig, gchar subst)
{
	register gchar *p = str;

	while (*p) {
		if (*p == orig)
			*p = subst;
		p++;
	}
}

gboolean is_header_line(const gchar *str)
{
	if (*str == ':') return FALSE;

	while (*str != '\0' && *str != ' ') {
		if (*str == ':' && *(str + 1) == ' ')
			return TRUE;
		str++;
	}

	return FALSE;
}

gchar *get_rc_dir(void)
{
	static gchar *rc_dir = NULL;

	if (!rc_dir)
		rc_dir = g_strconcat(g_get_home_dir(), G_DIR_SEPARATOR_S,
				     RC_DIR, NULL);

	return rc_dir;
}

gchar *get_news_cache_dir(void)
{
	static gchar *news_cache_dir = NULL;

	if (!news_cache_dir)
		news_cache_dir = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S,
					     NEWS_CACHE_DIR, NULL);

	return news_cache_dir;
}

gchar *get_domain_name(void)
{
	static gchar *domain_name = NULL;

	if (!domain_name) {
		gchar buf[BUFFSIZE] = "";
		gint val;

		if ((val = getdomainname(buf, sizeof(buf))) < 0)
			perror("getdomainname");
		if (val < 0 || buf[0] == '(' || buf[0] == '\0') {
			if (gethostname(buf, sizeof(buf)) < 0) {
				perror("gethostname");
				strcpy(buf, "unknown");
			}
		}

		domain_name = g_strdup(buf);
	}

	return domain_name;
}

off_t get_file_size(const gchar *file)
{
	struct stat s;

	if (stat(file, &s) < 0) {
		fprintf(stderr, "%s: ", file);
		perror("stat");
		return -1;
	}

	return s.st_size;
}

gboolean is_file_exist(const gchar *file)
{
	struct stat s;

	if (stat(file, &s) < 0) {
		if (ENOENT != errno) {
			fprintf(stderr, "%s: ", file);
			perror("stat");
		}
		return FALSE;
	}

	if (S_ISREG(s.st_mode))
		return TRUE;

	return FALSE;
}

gboolean is_dir_exist(const gchar *dir)
{
	struct stat s;

	if (stat(dir, &s) < 0) {
		//fprintf(stderr, "%s: ", dir);
		//perror("stat");
		return FALSE;
	}

	if (S_ISDIR(s.st_mode))
		return TRUE;

	return FALSE;
}

gint change_dir(const gchar *dir)
{
	gchar *prevdir = NULL;

	if (debug_mode)
		prevdir = g_get_current_dir();

	if (chdir(dir) < 0) {
		fprintf(stderr, "%s: ", dir);
		perror("chdir");
		if (debug_mode);
			g_free(prevdir);
		return -1;
	} else if (debug_mode) {
		gchar *cwd;

		cwd = g_get_current_dir();
		if (strcmp(prevdir, cwd) != 0)
			g_print("current dir: %s\n", cwd);
		g_free(cwd);
		g_free(prevdir);
	}

	return 0;
}

gint remove_dir_recursive(const gchar *dir)
{
	struct stat s;
	DIR *dp;
	struct dirent *d;
	gchar *prev_dir;

	//g_print("dir = %s\n", dir);

	if (stat(dir, &s) < 0) {
		g_warning("can't stat %s\n", dir);
		perror("stat");
		if (ENOENT == errno) return 0;
		return -1;
	}

	if (!S_ISDIR(s.st_mode)) {
		if (unlink(dir) < 0) {
			fprintf(stderr, "%s: ", dir);
			perror("unlink");
			return -1;
		}

		return 0;
	}

	prev_dir = g_get_current_dir();
	//g_print("prev_dir = %s\n", prev_dir);

	if (chdir(dir) < 0) {
		fprintf(stderr, "%s: ", dir);
		perror("chdir");
		g_free(prev_dir);
		return -1;
	}

	if ((dp = opendir(".")) == NULL) {
		perror("opendir");
		chdir(prev_dir);
		g_free(prev_dir);
		return -1;
	}

	/* remove all files in the directory */
	while ((d = readdir(dp)) != NULL) {
		if (!strcmp(d->d_name, ".") ||
		    !strcmp(d->d_name, ".."))
			continue;

		if (stat(d->d_name, &s) < 0) {
			fprintf(stderr, "%s: ", d->d_name);
			perror("stat");
			continue;
		}

		//g_print("removing %s\n", d->d_name);

		if (S_ISDIR(s.st_mode)) {
			if (remove_dir_recursive(d->d_name) < 0) {
				g_warning("can't remove directory\n");
				return -1;
			}
		} else {
			if (unlink(d->d_name) < 0) {
				fprintf(stderr, "%s: ", d->d_name);
				perror("unlink");
			}
		}
	}

	closedir(dp);

	if (chdir(prev_dir) < 0) {
		fprintf(stderr, "%s: ", prev_dir);
		perror("chdir");
		g_free(prev_dir);
		return -1;
	}

	g_free(prev_dir);

	if (rmdir(dir) < 0) {
		fprintf(stderr, "%s: ", dir);
		perror("rmdir");
		return -1;
	}

	return 0;
}

gint copy_file(const gchar *src, const gchar *dest)
{
	FILE *src_fp, *dest_fp;
	gchar buf[BUFSIZ];
	gchar *dest_bak = NULL;

	if ((src_fp = fopen(src, "r")) == NULL) {
		fprintf(stderr, "%s: ", src);
		perror("fopen");
		return -1;
	}
	if (is_file_exist(dest)) {
		dest_bak = g_strconcat(dest, ".bak", NULL);
		if (rename(dest, dest_bak) < 0) {
			fprintf(stderr, "%s: ", dest);
			perror("rename");
			fclose(src_fp);
			g_free(dest_bak);
			return -1;
		}
	}
	if ((dest_fp = fopen(dest, "w")) == NULL) {
		fprintf(stderr, "%s: ", dest);
		perror("fopen");
		fclose(src_fp);
		if (dest_bak) {
			if (rename(dest_bak, dest) < 0) {
				fprintf(stderr, "%s: ", dest_bak);
				perror("rename");
			}
			g_free(dest_bak);
		}
		return -1;
	}

	if (chmod(dest, S_IRUSR|S_IWUSR) < 0) {
		fprintf(stderr, "%s: ", dest);
		perror("chmod");
		g_warning(_("can't change file mode\n"));
	}

	while (fgets(buf, sizeof(buf), src_fp) != NULL) {
		if (fputs(buf, dest_fp) == EOF) {
			g_warning(_("writing to %s failed.\n"), dest);
			fclose(dest_fp);
			fclose(src_fp);
			unlink(dest);
			if (dest_bak) {
				if (rename(dest_bak, dest) < 0) {
					fprintf(stderr, "%s: ", dest_bak);
					perror("rename");
				}
				g_free(dest_bak);
			}
			return -1;
		}
	}

	fclose(dest_fp);
	fclose(src_fp);
	g_free(dest_bak);
	return 0;
}

void debug_print(const gchar *format, ...)
{
	va_list args;
	gchar buf[BUFFSIZE];

	if (!debug_mode) return;

	va_start(args, format);
	g_vsnprintf(buf, sizeof(buf), format, args);
	va_end(args);

	fputs(buf, stdout);
}

void log_print(const gchar *format, ...)
{
	va_list args;
	gchar buf[BUFFSIZE];

	va_start(args, format);
	g_vsnprintf(buf, sizeof(buf), format, args);
	va_end(args);

	if (debug_mode) fputs(buf, stdout);
	log_window_append(buf);
	statusbar_puts_all(buf);
}
